import {
    AlertDialog,
    AlertDialogContent,
    AlertDialogDescription,
    AlertDialogHeader,
    AlertDialogTitle,
} from "@/Components/ui/alert-dialog";
import { Button } from "@/Components/ui/button";
import { Input } from "@/Components/ui/input";
import { Label } from "@/Components/ui/label";
import { useForm } from "@inertiajs/react";
import { FormEventHandler, useId } from "react";
import { useVideoStore } from "@/Store/useVideoStore";
import { toast } from "sonner";
import { Page } from "@inertiajs/core";
import { validateInputText } from "@/Utils/validations";
import { useState } from "react";

export function ModalCreateOrEdit() {
    const idkey = useId();
    const { video, openModalCreateOrEditVideo, setOpenModalCreateOrEditVideo } =
        useVideoStore();

    const { data, setData, post, put, reset, clearErrors, processing, errors } =
        useForm<{
            id: number | null;
            key: string;
        }>( {
            id: video?.id ?? null,
            key: video?.key || "",
        });

    const [showIframe, setShowIframe] = useState(false);

    const submit: FormEventHandler = (e) => {
        e.preventDefault();

        const options = {
            preserveScroll: true,
            onSuccess: (response: Page<{ status?: { success?: string } }>) => {
                reset();
                const successMessage =
                    response.props.status?.success ||
                    "Operación realizada con éxito.";
                toast.success(successMessage);
                setOpenModalCreateOrEditVideo(false);
            },
            onError: (response: { key?: string }) => {
                toast.error(response.key || "Error en el formulario.");
            },
        };

        if (data.id) {
            put(route("videos.update", { video: data.id }), options);
        } else {
            post(route("videos.store"), options);
        }
        clearErrors();
    };

    const handleNameChange = (e: React.ChangeEvent<HTMLInputElement>) => {
        const value = e.target.value;
        if (validateInputText(value)) {
            setData("key", value); 
        }
    };

    const showVideo = () => {
        setShowIframe(true); 
    };

    return (
        <AlertDialog
            open={openModalCreateOrEditVideo}
            onOpenChange={() => setOpenModalCreateOrEditVideo(false)}
        >
            <AlertDialogContent>
                <AlertDialogHeader>
                    <AlertDialogTitle>
                        {video?.id ? "Editar" : "Crear"} Video
                    </AlertDialogTitle>
                    <AlertDialogDescription>
                        {video?.id
                            ? "Modifica los datos del video y guarda los cambios."
                            : "Ingresa los datos para registrar un nuevo video."}
                    </AlertDialogDescription>
                </AlertDialogHeader>
                <form onSubmit={submit} className="overflow-y-auto max-h-[80vh]">
                    <Label htmlFor={idkey}>key</Label>
                    <Input
                        id={idkey}
                        type="text"
                        value={data.key}
                        className="mt-1 block w-full"
                        placeholder="key del video"
                        onChange={handleNameChange}
                        required
                    />
                    {errors.key && (
                        <Label className="text-destructive" htmlFor={idkey}>
                            {errors.key}
                        </Label>
                    )}
                    { data.key &&( <Button type="button" onClick={showVideo} className="w-full my-4">
                        Mostrar Video
                    </Button>) }
                   
                    {showIframe && data.key && (
                       <iframe
                            src={`https://www.tiktok.com/embed/${data.key}?reels=0&no_ads=1&autoplay=1&hide_related=1&showinfo=0&controls=0&loop=1&muted=1`}
                            width="100%"
                            height="500"
                            frameBorder="0"
                            allowFullScreen
                            title="Video de TikTok"
                        />
                    )}

                    <footer className="flex justify-around gap-4 mt-4">
                        <Button
                            type="button"
                            onClick={() => setOpenModalCreateOrEditVideo(false)}
                            variant="secondary"
                        >
                            Cancelar
                        </Button>
                        <Button type="submit" disabled={processing}>
                            Guardar
                        </Button>
                    </footer>
                </form>
            </AlertDialogContent>
        </AlertDialog>
    );
}
