import { Pagination } from "./Components/Pagination";
import SearchBar from "@/Components/SearchBar";
import { Button } from "@/Components/ui/button";
import {
    Card,
    CardContent,
    CardDescription,
    CardFooter,
    CardHeader,
    CardTitle,
} from "@/Components/ui/card";
import LayoutPrivate from "@/Layouts/Private/layout";
import { useAsistenciaStore } from "@/Store/useAsistenciaStore";
import { useConductaStore } from "@/Store/useConductaStore";
import { useNotaStore } from "@/Store/useNotaStore";
import { Paginacion } from "@/types/param";
import { router, usePage } from "@inertiajs/react";
import { CirclePlus } from "lucide-react";
import { useState } from "react";
import DataTableNota from "./Notas/Components/DataTable";
import { ModalDeleteNota } from "./Notas/Components/ModalDelete";
import { ModalCreateOrEditNota } from "./Notas/Components/ModalCreateOrEdit";
import { ModalCreateOrEditAsistencia } from "./Asistencias/Components/ModalCreateOrEdit";
import { ModalDeleteAsistencia } from "./Asistencias/Components/ModalDelete";
import { ModalCreateOrEditConducta } from "./Conductas/Components/ModalCreateOrEdit";
import { ModalDeleteConducta } from "./Conductas/Components/ModalDelete";
import DataTableConducta from "./Conductas/Components/DataTable";
import DataTableAsistencia from "./Asistencias/Components/DataTable";
import { Estudiante } from "@/types/estudiante";

interface Filtro {
    search?: string;
    per_page?: number;
    page?: number;
}

interface Registro {
    id: number;
    fecha: string;
    descripcion?: string;
}

interface Promedio {
    notas: Registro[];
    estudiante: Estudiante;
    conductas: Registro[];
    asistencias: Registro[];
    filters: {
        nota: Filtro;
        conducta: Filtro;
        asistencia: Filtro;
    };
    status?: {
        success?: string;
        error?: string;
    };
}

interface PageProps<T> {
    data: Paginacion<T>;
    filters: {
        nota: Filtro;
        conducta: Filtro;
        asistencia: Filtro;
    };
    [key: string]: any;
}

export default function Index() {
    const { auth, estudiante, notas, conductas, asistencias, filters } =
        usePage<PageProps<Promedio>>().props;

    const [searchNota, setSearchNota] = useState<string>(
        filters.nota?.search || ""
    );
    const [perPageNota, setPerPageNota] = useState<number>(
        filters.nota?.per_page || 10
    );
    const [pageNota, setPageNota] = useState<number>(
        filters.nota?.page || 1
    );

    const [searchAsistencia, setSearchAsistencia] = useState<string>(
        filters.asistencia?.search || ""
    );
    const [perPageAsistencia, setPerPageAsistencia] = useState<number>(
        filters.asistencia?.per_page || 10
    );
    const [pageAsistencia, setPageAsistencia] = useState<number>(
        filters.asistencia?.page || 1
    );

    const [searchConducta, setSearchConducta] = useState<string>(
        filters.conducta?.search || ""
    );
    const [perPageConducta, setPerPageConducta] = useState<number>(
        filters.conducta?.per_page || 10
    );
    const [pageConducta, setPageConducta] = useState<number>(
        filters.conducta?.page || 1
    );

    const {
        setNota,
        openModalCreateOrEditNota,
        openModalDeleteNota,
        setOpenModalCreateOrEditNota,
    } = useNotaStore();

    const {
        setAsistencia,
        openModalCreateOrEditAsistencia,
        openModalDeleteAsistencia,
        setOpenModalCreateOrEditAsistencia,
    } = useAsistenciaStore();

    const {
        setConducta,
        openModalCreateOrEditConducta,
        openModalDeleteConducta,
        setOpenModalCreateOrEditConducta,
    } = useConductaStore();

    const id_estudiante = usePage<PageProps<Promedio>>().props.estudiante?.id;

    const handlePerPageChange = (
        value: string,
        type: "nota" | "asistencia" | "conducta"
    ) => {
        const perPageValue = Number(value);

        router.get(
            route("promedios.index"),
            {
                id_estudiante,
                [type]: {
                    search:
                        type === "nota"
                            ? searchNota
                            : type === "asistencia"
                                ? searchAsistencia
                                : searchConducta,
                    per_page: perPageValue,
                },
            },
            { preserveState: true }
        );

        if (type === "nota") {
            setPerPageNota(perPageValue);
        } else if (type === "asistencia") {
            setPerPageAsistencia(perPageValue);
        } else if (type === "conducta") {
            setPerPageConducta(perPageValue);
        }
    };

    const handleSearch = (
        value: string,
        type: "nota" | "asistencia" | "conducta"
    ) => {
        router.get(
            route("promedios.index"),
            {
                id_estudiante,
                [type]: {
                    search: value,
                    per_page:
                        type === "nota"
                            ? perPageNota
                            : type === "asistencia"
                                ? perPageAsistencia
                                : perPageConducta,
                },
            },
            { preserveState: true }
        );

        if (type === "nota") {
            setSearchNota(value);
        } else if (type === "asistencia") {
            setSearchAsistencia(value);
        } else if (type === "conducta") {
            setSearchConducta(value);
        }
    };


    const handlePageChange = (newPage: number, type: "nota" | "asistencia" | "conducta") => {

        const lastPage =
            type === "nota"
                ? notas.last_page
                : type === "asistencia"
                    ? asistencias.last_page
                    : conductas.last_page;

        if (newPage >= 1 && newPage <= lastPage) {
            const searchValue = type === "nota" ? searchNota : type === "asistencia" ? searchAsistencia : searchConducta;
            const perPageValue = type === "nota" ? perPageNota : type === "asistencia" ? perPageAsistencia : perPageConducta;

            router.get(
                route("promedios.index"),
                {
                    id_estudiante,
                    [type]: {
                        search: searchValue,
                        per_page: perPageValue,
                        page: newPage,
                    },
                },
                { preserveState: true }
            );

            if (type === "nota") {
                setPageNota(newPage);
            } else if (type === "asistencia") {
                setPageAsistencia(newPage);
            } else if (type === "conducta") {
                setPageConducta(newPage);
            }
        }
    };


    return (
        <LayoutPrivate user={auth.user} header={`${estudiante.nombres} ${estudiante.apellido_paterno} ${estudiante.apellido_materno}`}>
            <div className="w-full max-w-6xl">
                <Card className="mb-4">
                    <CardHeader>
                        <CardTitle className="text-center pb-3">
                            Notas
                        </CardTitle>
                        <CardDescription className="flex gap-4">
                            <SearchBar
                                className="flex-1"
                                keyName="Nota"
                                search={searchNota}
                                perPage={perPageNota}
                                onSearch={(value) =>
                                    handleSearch(value, "nota")
                                }
                                onPerPageChange={(value) =>
                                    handlePerPageChange(value, "nota")
                                }
                            />
                            <Button
                                onClick={() => {
                                    setOpenModalCreateOrEditNota(true);
                                    setNota(undefined);
                                }}
                            >
                                Crear <CirclePlus />
                            </Button>
                        </CardDescription>
                    </CardHeader>
                    <CardContent>
                        <DataTableNota data={notas.data} />
                    </CardContent>
                    <CardFooter className="justify-center">
                        <Pagination
                            currentPage={notas.current_page}
                            lastPage={notas.last_page}
                            prevPageUrl={notas.prev_page_url}
                            nextPageUrl={notas.next_page_url}
                            search={searchNota}
                            perPage={perPageNota}
                            onPageChange={(newPage: number) => handlePageChange(newPage, "nota")}
                        />
                    </CardFooter>
                </Card>

                <Card className="mb-4">
                    <CardHeader>
                        <CardTitle className="text-center pb-3">
                            Asistencias
                        </CardTitle>
                        <CardDescription className="flex gap-4">
                            <SearchBar
                             keyName="Asistencia"
                                className="flex-1"
                                search={searchAsistencia}
                                perPage={perPageAsistencia}
                                onSearch={(value) =>
                                    handleSearch(value, "asistencia")
                                }
                                onPerPageChange={(value) =>
                                    handlePerPageChange(value, "asistencia")
                                }
                            />
                            <Button
                                onClick={() => {
                                    setOpenModalCreateOrEditAsistencia(true);
                                    setAsistencia(undefined);
                                }}
                            >
                                Crear <CirclePlus />
                            </Button>
                        </CardDescription>
                    </CardHeader>
                    <CardContent>
                        <DataTableAsistencia data={asistencias.data} />
                    </CardContent>
                    <CardFooter className="justify-center">
                        <Pagination
                            currentPage={asistencias.current_page}
                            lastPage={asistencias.last_page}
                            prevPageUrl={asistencias.prev_page_url}
                            nextPageUrl={asistencias.next_page_url}
                            search={searchAsistencia}
                            perPage={perPageAsistencia}
                            onPageChange={(newPage: number) => handlePageChange(newPage, "asistencia")}
                        />
                    </CardFooter>
                </Card>

                <Card className="mb-4">
                    <CardHeader>
                        <CardTitle className="text-center pb-3">
                            Conductas
                        </CardTitle>
                        <CardDescription className="flex gap-4">
                            <SearchBar
                             keyName="Conducta"
                                className="flex-1"
                                search={searchConducta}
                                perPage={perPageConducta}
                                onSearch={(value) =>
                                    handleSearch(value, "conducta")
                                }
                                onPerPageChange={(value) =>
                                    handlePerPageChange(value, "conducta")
                                }
                            />
                            <Button
                                onClick={() => {
                                    setOpenModalCreateOrEditConducta(true);
                                    setConducta(undefined);
                                }}
                            >
                                Crear <CirclePlus />
                            </Button>
                        </CardDescription>
                    </CardHeader>
                    <CardContent>
                        <DataTableConducta data={conductas.data} />
                    </CardContent>
                    <CardFooter className="justify-center">
                        <Pagination
                            currentPage={conductas.current_page}
                            lastPage={conductas.last_page}
                            prevPageUrl={conductas.prev_page_url}
                            nextPageUrl={conductas.next_page_url}
                            search={searchConducta}
                            perPage={perPageConducta}
                            onPageChange={(newPage: number) => handlePageChange(newPage, "conducta")}
                        />
                    </CardFooter>
                </Card>
            </div>

            {openModalCreateOrEditNota && <ModalCreateOrEditNota />}
            {openModalDeleteNota && <ModalDeleteNota />}
            {openModalCreateOrEditAsistencia && <ModalCreateOrEditAsistencia />}
            {openModalDeleteAsistencia && <ModalDeleteAsistencia />}
            {openModalCreateOrEditConducta && <ModalCreateOrEditConducta />}
            {openModalDeleteConducta && <ModalDeleteConducta />}
        </LayoutPrivate>
    );
}
