import {
    AlertDialog,
    AlertDialogContent,
    AlertDialogDescription,
    AlertDialogHeader,
    AlertDialogTitle,
} from "@/Components/ui/alert-dialog";
import { Button } from "@/Components/ui/button";
import { Input } from "@/Components/ui/input";
import { Label } from "@/Components/ui/label";
import { useForm } from "@inertiajs/react";
import { FormEventHandler, useId } from "react";
import { useGradoStore } from "@/Store/useGradoStore";
import { toast } from "sonner";
import { Page } from "@inertiajs/core";
import { validateInputText } from "@/Utils/validations";

export function ModalCreateOrEdit() {
    const idNombre = useId();
    const { grado, openModalCreateOrEditGrado, setOpenModalCreateOrEditGrado } =
        useGradoStore();

    const { data, setData, post, put, reset, clearErrors, processing, errors } =
        useForm<{
            id: number | null;
            nombre: string;
        }>({
            id: grado?.id ?? null,
            nombre: grado?.nombre || "",
        });

    const submit: FormEventHandler = (e) => {
        e.preventDefault();

        const options = {
            preserveScroll: true,
            onSuccess: (response: Page<{ status?: { success?: string } }>) => {
                reset();
                const successMessage =
                    response.props.status?.success ||
                    "Operación realizada con éxito.";
                toast.success(successMessage);
                setOpenModalCreateOrEditGrado(false);
            },
            onError: (response: { nombre?: string }) => {
                toast.error(response.nombre || "Error en el formulario.");
            },
        };

        if (data.id) {
            put(route("grados.update", { grado: data.id }), options);
        } else {
            post(route("grados.store"), options);
        }
        clearErrors();
    };

    const handleNameChange = (e: React.ChangeEvent<HTMLInputElement>) => {
        const value = e.target.value;
        if (validateInputText(value)) {
            setData("nombre", value); 
        }
    };

    return (
        <AlertDialog
            open={openModalCreateOrEditGrado}
            onOpenChange={() => setOpenModalCreateOrEditGrado(false)}
        >
            <AlertDialogContent>
                <AlertDialogHeader>
                    <AlertDialogTitle>
                        {grado?.id ? "Editar" : "Crear"} Grado Escolar
                    </AlertDialogTitle>
                    <AlertDialogDescription>
                        {grado?.id
                            ? "Modifica los datos del grado y guarda los cambios."
                            : "Ingresa los datos para registrar un nuevo grado."}
                    </AlertDialogDescription>
                </AlertDialogHeader>
                <form onSubmit={submit}>
                    <Label htmlFor={idNombre}>Nombre</Label>
                    <Input
                        id={idNombre}
                        type="text"
                        value={data.nombre}
                        className="mt-1 block w-full"
                        placeholder="Nombre del grado"
                        onChange={handleNameChange}
                        required
                    />
                    {errors.nombre && (
                        <Label className="text-destructive" htmlFor={idNombre}>
                            {errors.nombre}
                        </Label>
                    )}
                    <footer className="flex justify-around gap-4 mt-4">
                        <Button
                            type="button"
                            onClick={() => setOpenModalCreateOrEditGrado(false)}
                            variant="secondary"
                        >
                            Cancelar
                        </Button>
                        <Button type="submit" disabled={processing}>
                            Guardar
                        </Button>
                    </footer>
                </form>
            </AlertDialogContent>
        </AlertDialog>
    );
}
