import {
    AlertDialog,
    AlertDialogContent,
    AlertDialogDescription,
    AlertDialogHeader,
    AlertDialogTitle,
} from "@/Components/ui/alert-dialog";
import { Button } from "@/Components/ui/button";
import { Label } from "@/Components/ui/label";
import { Input } from "@/Components/ui/input";
import { useForm, usePage } from "@inertiajs/react";
import { FormEventHandler, useId, useState } from "react";
import { useCursoMaterialStore } from "@/Store/useCursoMaterialStore";
import { toast } from "sonner";
import { Page } from "@inertiajs/core";
import { PageProps } from "@/types/param";
import {
    Select,
    SelectContent,
    SelectItem,
    SelectTrigger,
    SelectValue,
} from "@/Components/ui/select";
import { FileInput } from "@/Components/FileInput";
import { X } from "lucide-react";
import { CursoMaterial } from "@/types/curso-material";
import { Curso } from "@/types/curso";
import { validateInputText } from "@/Utils/validations";

export function ModalCreateOrEdit() {
    const idNombre = useId();
    const {
        cursoMaterial,
        openModalCreateOrEditCursoMaterial,
        setOpenModalCreateOrEditCursoMaterial,
    } = useCursoMaterialStore();

    const { cursos } = usePage<PageProps<CursoMaterial>>().props;

    const { data, setData, post, put, reset, clearErrors, processing, errors } =
        useForm<{
            id: number | undefined;
            nombre: string;
            curso: Curso | null;
            videos: File[];
            imagenes: File[];
            pdfs: File[];
        }>({
            id: cursoMaterial?.id || undefined,
            nombre: cursoMaterial?.nombre || "",
            curso: cursoMaterial?.curso || null,
            videos: [],
            imagenes: [],
            pdfs: [],
        });

    const [previewVideos, setPreviewVideos] = useState<string[]>([]);
    const [previewImagenes, setPreviewImagenes] = useState<string[]>([]);
    const [previewPdfs, setPreviewPdfs] = useState<string[]>([]);

    const handleFileChange = (
        type: "videos" | "imagenes" | "pdfs",
        files: FileList | null
    ) => {
        if (!files) return;

        const fileArray = Array.from(files);

        setData(type, [...data[type], ...fileArray]);

        if (type === "videos") {
            const newPreviews = fileArray.map((file) =>
                URL.createObjectURL(file)
            );
            setPreviewVideos([...previewVideos, ...newPreviews]);
        } else if (type === "imagenes") {
            const newPreviews = fileArray.map((file) =>
                URL.createObjectURL(file)
            );
            setPreviewImagenes([...previewImagenes, ...newPreviews]);
        } else if (type === "pdfs") {
            const newPreviews = fileArray.map((file) => file.name);
            setPreviewPdfs([...previewPdfs, ...newPreviews]);
        }
    };

    const removeFile = (
        type: "videos" | "imagenes" | "pdfs",
        index: number
    ) => {
        const newFiles = [...data[type]];
        newFiles.splice(index, 1);
        setData(type, newFiles);

        if (type === "videos") {
            const newPreviews = [...previewVideos];
            URL.revokeObjectURL(newPreviews[index]);
            newPreviews.splice(index, 1);
            setPreviewVideos(newPreviews);
        } else if (type === "imagenes") {
            const newPreviews = [...previewImagenes];
            URL.revokeObjectURL(newPreviews[index]);
            newPreviews.splice(index, 1);
            setPreviewImagenes(newPreviews);
        } else if (type === "pdfs") {
            const newPreviews = [...previewPdfs];
            newPreviews.splice(index, 1);
            setPreviewPdfs(newPreviews);
        }
    };

    const submit: FormEventHandler = (e) => {
        e.preventDefault();

        const formData = new FormData();

        if (data.curso) {
            formData.append("curso_id", data.curso.id.toString());
        }

        data.videos.forEach((file, index) => {
            formData.append(`videos[${index}]`, file);
        });

        data.imagenes.forEach((file, index) => {
            formData.append(`imagenes[${index}]`, file);
        });

        data.pdfs.forEach((file, index) => {
            formData.append(`pdfs[${index}]`, file);
        });

        const options = {
            preserveScroll: true,
            onSuccess: (response: Page<{ status?: { success?: string } }>) => {
                reset();
                const successMessage =
                    response.props.status?.success ||
                    "Operación realizada con éxito.";
                toast.success(successMessage);
                setOpenModalCreateOrEditCursoMaterial(false);

                previewVideos.forEach((url) => URL.revokeObjectURL(url));
                previewImagenes.forEach((url) => URL.revokeObjectURL(url));
                setPreviewVideos([]);
                setPreviewImagenes([]);
                setPreviewPdfs([]);
            },

            onError: (errors: Record<string, string>) => {
                if (errors) {
                    console.error("errors ", errors);
                    Object.entries(errors).forEach(([field, message]) => {
                        toast.error(`${field}: ${message}`);
                    });
                } else {
                    toast.error("Error en el formulario.");
                }
            },
        };

        if (cursoMaterial?.id) {
            put(
                route("cursos-materiales.update", {
                    cursos_materiale: cursoMaterial?.id,
                }),
                {
                    ...options,
                    data: formData,
                }
            );
        } else {
            post(route("cursos-materiales.store"), {
                ...options,
                data: formData,
            });
        }
    };

    const handleNameChange = (e: React.ChangeEvent<HTMLInputElement>) => {
        const value = e.target.value;
        if (validateInputText(value)) {
            setData("nombre", value);
        }
    };
   
    return (
        <AlertDialog
            open={openModalCreateOrEditCursoMaterial}
            onOpenChange={() => setOpenModalCreateOrEditCursoMaterial(false)}
        >
            <AlertDialogContent>
                <AlertDialogHeader>
                    <AlertDialogTitle>Sesión de Estudio</AlertDialogTitle>
                    <AlertDialogDescription>
                        Sube los archivos para registrar o actualizar la sesión
                        de estudio.
                    </AlertDialogDescription>
                </AlertDialogHeader>
                <form onSubmit={submit} encType="multipart/form-data">
                    <div className="mb-4">
                        <Label htmlFor={idNombre}>Nombre de Sesión</Label>
                        <Input
                            id={idNombre}
                            type="text"
                            value={data.nombre}
                            className="mt-1 block w-full"
                            placeholder="Nombre de Sesión"
                            onChange={handleNameChange}
                            required
                        />

                        <Label htmlFor="curso?.id">Curso</Label>
                        <Select
                            value={String(data.curso?.id) || undefined}
                            onValueChange={(value) => {
                                const selectedCurso: Curso | undefined =
                                    cursos.find(
                                        (c: Curso) => c.id.toString() === value
                                    );
                                if (selectedCurso) {
                                    setData("curso", selectedCurso);
                                } else {
                                    setData("curso", null);
                                }
                            }}
                        >
                            <SelectTrigger className="w-full">
                                <SelectValue placeholder="Seleccione un curso" />
                            </SelectTrigger>
                            <SelectContent>
                                {cursos.length > 0 ? (
                                    cursos.map((curso: Curso) => (
                                        <SelectItem
                                            key={curso.id}
                                            value={curso.id.toString()}
                                            defaultValue={cursoMaterial?.curso?.id.toString()}
                                        >
                                            {curso.nombre}
                                        </SelectItem>
                                    ))
                                ) : (
                                    <SelectItem disabled value="">
                                        No hay cursos disponibles
                                    </SelectItem>
                                )}
                            </SelectContent>
                        </Select>
                        {errors.curso && (
                            <p className="mt-1 text-sm text-destructive">
                                {errors.curso.toString()}
                            </p>
                        )}
                    </div>

                    <div className="mb-4">
                        <Label>Videos</Label>
                        <FileInput
                            id="videos"
                            accept="video/mp4,video/avi,video/mpeg,video/quicktime"
                            multiple
                            onChange={(e) =>
                                handleFileChange("videos", e.target.files)
                            }
                        />
                        {errors.videos && (
                            <p className="mt-1 text-sm text-destructive">
                                {errors.videos}
                            </p>
                        )}

                        <div className="mt-2 space-y-2">
                            {previewVideos.map((preview, index) => (
                                <div
                                    key={index}
                                    className="flex items-center justify-between p-2 border rounded"
                                >
                                    <video
                                        src={preview}
                                        className="h-20 rounded"
                                        controls
                                    />
                                    <Button
                                        type="button"
                                        variant="ghost"
                                        size="icon"
                                        onClick={() =>
                                            removeFile("videos", index)
                                        }
                                    >
                                        <X className="h-4 w-4" />
                                    </Button>
                                </div>
                            ))}
                        </div>
                    </div>

                    <div className="mb-4">
                        <Label>Imágenes</Label>
                        <FileInput
                            id="imagenes"
                            accept="image/jpeg,image/png,image/jpg"
                            multiple
                            onChange={(e) =>
                                handleFileChange("imagenes", e.target.files)
                            }
                        />
                        {errors.imagenes && (
                            <p className="mt-1 text-sm text-destructive">
                                {errors.imagenes}
                            </p>
                        )}

                        <div className="mt-2 grid grid-cols-3 gap-2">
                            {previewImagenes.map((preview, index) => (
                                <div key={index} className="relative group">
                                    <img
                                        src={preview}
                                        alt={`Preview ${index}`}
                                        className="h-20 w-full object-cover rounded"
                                    />
                                    <Button
                                        type="button"
                                        variant="ghost"
                                        size="icon"
                                        className="absolute top-1 right-1 opacity-0 group-hover:opacity-100 transition-opacity"
                                        onClick={() =>
                                            removeFile("imagenes", index)
                                        }
                                    >
                                        <X className="h-4 w-4" />
                                    </Button>
                                </div>
                            ))}
                        </div>
                    </div>

                    <div className="mb-4">
                        <Label>PDFs</Label>
                        <FileInput
                            id="pdfs"
                            accept=".pdf"
                            multiple
                            onChange={(e) =>
                                handleFileChange("pdfs", e.target.files)
                            }
                        />
                        {errors.pdfs && (
                            <p className="mt-1 text-sm text-destructive">
                                {errors.pdfs}
                            </p>
                        )}

                        <div className="mt-2 space-y-1">
                            {previewPdfs.map((name, index) => (
                                <div
                                    key={index}
                                    className="flex items-center justify-between p-2 border rounded"
                                >
                                    <span className="truncate">{name}</span>
                                    <Button
                                        type="button"
                                        variant="ghost"
                                        size="icon"
                                        onClick={() =>
                                            removeFile("pdfs", index)
                                        }
                                    >
                                        <X className="h-4 w-4" />
                                    </Button>
                                </div>
                            ))}
                        </div>
                    </div>

                    <footer className="flex justify-around gap-4 mt-4">
                        <Button
                            type="button"
                            onClick={() =>
                                setOpenModalCreateOrEditCursoMaterial(false)
                            }
                            variant="secondary"
                        >
                            Cancelar
                        </Button>
                        <Button type="submit" disabled={processing}>
                            {processing ? "Guardando..." : "Guardar"}
                        </Button>
                    </footer>
                </form>
            </AlertDialogContent>
        </AlertDialog>
    );
}
