import { Button } from "@/Components/ui/button";
import { Label } from "@/Components/ui/label";
import { Input } from "@/Components/ui/input";
import { useForm, usePage } from "@inertiajs/react";
import { PageProps } from "@/types/param";
import {
    Select,
    SelectContent,
    SelectItem,
    SelectTrigger,
    SelectValue,
} from "@/Components/ui/select";
import { FileText, X } from "lucide-react";
import {
    CursoMaterial,
    CursoMaterialImagen,
    CursoMaterialVideo,
} from "@/types/curso-material";
import { Curso } from "@/types/curso";
import LayoutPrivate from "@/Layouts/Private/layout";
import { FormEventHandler, useEffect, useId, useRef, useState } from "react";
import { toast } from "sonner";
import { validateInputText } from "@/Utils/validations";
import { TikTokEmbed } from "@/Components/TikTokEmbed";

interface Red {
    id: number;
    key: string;
}

type FormData = {
    id: number;
    nombre: string;
    id_curso: string;
    deleted_video_ids: number[];
    deleted_imagen_ids: number[];
    deleted_pdf_ids: number[];
    videos: File[];
    imagenes: File[];
    pdfs: File[];
    redes: Red[];
};

export default function Edit() {
    const idNombre = useId();
    const { curso_material, auth, cursos } =
        usePage<PageProps<CursoMaterial>>().props;

    const videoInputRef = useRef<HTMLInputElement>(null);
    const imagenInputRef = useRef<HTMLInputElement>(null);
    const pdfInputRef = useRef<HTMLInputElement>(null);

    // Preview states
    const [previewVideos, setPreviewVideos] = useState<any[]>(
        curso_material.cursos_materiales_videos.map((v: any) => ({
            type: "existing",
            id: v.id,
            url: `${v.url_video}`,
            name: v.original_name,
        }))
    );

    const [previewImagenes, setPreviewImagenes] = useState<any[]>(
        curso_material.cursos_materiales_imagenes.map((v: any) => ({
            type: "existing",
            id: v.id,
            url: `${v.url_imagen}`,
            name: v.original_name,
        }))
    );

    const [previewPdfs, setPreviewPdfs] = useState<any[]>(
        curso_material.cursos_materiales_pdfs.map((v: any) => ({
            type: "existing",
            id: v.id,
            url: `${v.url_pdf}`,
            name: v.original_name,
        }))
    );


    const { data, setData, post, reset, processing, errors } =
        useForm<FormData>({
            id: curso_material.id,
            nombre: curso_material.nombre,
            id_curso: curso_material.id_curso,
            deleted_video_ids: [],
            deleted_imagen_ids: [],
            deleted_pdf_ids: [],
            videos: [],
            imagenes: [],
            pdfs: [],
            redes: curso_material.cursos_materiales_redes.map((red: Red) => ({
                id: red.id,
                key: red.key
            })),

        });

    const handleVideoChange = (files: FileList | null) => {
        if (!files) return;

        const videoFiles = Array.from(files).filter((file) =>
            file.type.startsWith("video/")
        );

        if (videoFiles.length !== files.length) {
            toast.warning("Algunos archivos no son videos y fueron ignorados");
        }

        setData("videos", [...data.videos, ...videoFiles]);

        const newPreviews = videoFiles.map((file) => ({
            type: "new",
            url: URL.createObjectURL(file),
            name: file.name,
            file,
        }));

        setPreviewVideos([...previewVideos, ...newPreviews]);
    };

    const handleImagenChange = (files: FileList | null) => {
        if (!files) return;

        const imageFiles = Array.from(files).filter((file) =>
            file.type.startsWith("image/")
        );

        if (imageFiles.length !== files.length) {
            toast.warning(
                "Algunos archivos no son imágenes y fueron ignorados"
            );
        }

        setData("imagenes", [...data.imagenes, ...imageFiles]);

        const newPreviews = imageFiles.map((file) => ({
            type: "new",
            url: URL.createObjectURL(file),
            name: file.name,
            file,
        }));

        setPreviewImagenes([...previewImagenes, ...newPreviews]);
    };

    const handlePdfChange = (files: FileList | null) => {
        if (!files) return;

        const pdfFiles = Array.from(files).filter(
            (file) => file.type === "application/pdf"
        );

        if (pdfFiles.length !== files.length) {
            toast.warning("Algunos archivos no son PDFs y fueron ignorados");
        }

        setData("pdfs", [...data.pdfs, ...pdfFiles]);

        const newPreviews = pdfFiles.map((file) => ({
            type: "new",
            name: file.name,
            file,
        }));

        setPreviewPdfs([...previewPdfs, ...newPreviews]);
    };

    // Removal handlers
    const removeVideo = (index: number) => {
        const preview = previewVideos[index];
        if (preview.type === "existing") {
            setData("deleted_video_ids", [
                ...data.deleted_video_ids,
                preview.id!,
            ]);
        } else {
            // Remove from videos array
            const fileIndex = previewVideos
                .slice(0, index)
                .filter((p) => p.type === "new").length;
            const newFiles = [...data.videos];
            newFiles.splice(fileIndex, 1);
            setData("videos", newFiles);

            if (preview.url) URL.revokeObjectURL(preview.url);
        }
        setPreviewVideos(previewVideos.filter((_, i) => i !== index));
    };

    const removeImagen = (index: number) => {
        const preview = previewImagenes[index];
        if (preview.type === "existing") {
            setData("deleted_imagen_ids", [
                ...data.deleted_imagen_ids,
                preview.id!,
            ]);
        } else {
            // Remove from imagenes array
            const fileIndex = previewImagenes
                .slice(0, index)
                .filter((p) => p.type === "new").length;
            const newFiles = [...data.imagenes];
            newFiles.splice(fileIndex, 1);
            setData("imagenes", newFiles);

            if (preview.url) URL.revokeObjectURL(preview.url);
        }
        setPreviewImagenes(previewImagenes.filter((_, i) => i !== index));
    };

    const removePdf = (index: number) => {
        const preview = previewPdfs[index];
        if (preview.type === "existing") {
            setData("deleted_pdf_ids", [...data.deleted_pdf_ids, preview.id!]);
        } else {
            // Remove from pdfs array
            const fileIndex = previewPdfs
                .slice(0, index)
                .filter((p) => p.type === "new").length;
            const newFiles = [...data.pdfs];
            newFiles.splice(fileIndex, 1);
            setData("pdfs", newFiles);
        }
        setPreviewPdfs(previewPdfs.filter((_, i) => i !== index));
    };

    const submit: FormEventHandler = (e) => {
        e.preventDefault();

        const formData = new FormData();

        // Add basic data
        formData.append("id", data.id.toString());
        formData.append("nombre", data.nombre);
        formData.append("id_curso", data.id_curso);

        // Add arrays for deletion
        data.deleted_video_ids.forEach((id) =>
            formData.append("deleted_video_ids[]", id.toString())
        );
        data.deleted_imagen_ids.forEach((id) =>
            formData.append("deleted_imagen_ids[]", id.toString())
        );
        data.deleted_pdf_ids.forEach((id) =>
            formData.append("deleted_pdf_ids[]", id.toString())
        );

        // Add new files
        data.videos.forEach((file, i) => formData.append(`videos[${i}]`, file));
        data.imagenes.forEach((file, i) =>
            formData.append(`imagenes[${i}]`, file)
        );
        data.pdfs.forEach((file, i) => formData.append(`pdfs[${i}]`, file));

        post(route("cursos-materiales.update", data.id), {
            data: formData,
            preserveScroll: true,
            headers: {
                "Content-Type": "multipart/form-data",
            },
            onSuccess: () => {
                // Cleanup preview URLs
                [...previewVideos, ...previewImagenes].forEach((preview) => {
                    if (preview.type === "new" && preview.url) {
                        URL.revokeObjectURL(preview.url);
                    }
                });

                // Reset form
                reset();
                setPreviewVideos([]);
                setPreviewImagenes([]);
                setPreviewPdfs([]);

                toast.success("Material actualizado exitosamente");
            },
            onError: (errors) => {
                Object.entries(errors).forEach(([field, message]) => {
                    toast.error(`${field}: ${message}`);
                });
            },
        });
    };

    const [tikTokInput, setTikTokInput] = useState("");

    const addTikTokKey = () => {
        if (tikTokInput.trim()) {
            // Agregar la clave TikTok a las redes
            setData("redes", [
                ...data.redes,
                { id: Date.now(), key: tikTokInput.trim() }, // Se utiliza Date.now() para un ID único
            ]);
            setTikTokInput(""); // Limpiar el campo de entrada después de agregar
        }
    };


    // Función para eliminar un video de TikTok
    const removeTikTokKey = (index: number) => {
        // Eliminar el video de TikTok de la lista de redes
        const updatedRedes = data.redes.filter((_, i) => i !== index);
        setData("redes", updatedRedes);
    };


    return (
        <LayoutPrivate user={auth.user} header={"Sesiones de Estudio"}>
            <form
                onSubmit={submit}
                encType="multipart/form-data"
                className="w-full max-w-4xl mx-auto space-y-6"
            >
                {/* Nombre */}
                <div className="space-y-2">
                    <Label htmlFor={idNombre}>Nombre de Sesión</Label>
                    <Input
                        id={idNombre}
                        type="text"
                        value={data.nombre}
                        className="w-full"
                        placeholder="Nombre de Sesión"
                        onChange={(e) => setData("nombre", e.target.value)}
                        required
                    />
                    {errors.nombre && (
                        <p className="text-sm text-destructive">
                            {errors.nombre}
                        </p>
                    )}
                </div>
                {/* Curso */}
                <div className="space-y-2">
                    <Label>Curso</Label>
                    <Select
                        value={data.id_curso.toString()}
                        onValueChange={(value) => setData("id_curso", value)}
                        defaultValue={curso_material.id_curso.toString()}
                    >
                        <SelectTrigger className="w-full">
                            <SelectValue placeholder="Seleccione un curso" />
                        </SelectTrigger>
                        <SelectContent>
                            {cursos.map((curso: Curso) => (
                                <SelectItem
                                    key={curso.id}
                                    value={curso.id.toString()}
                                >
                                    {curso.nombre}
                                </SelectItem>
                            ))}
                        </SelectContent>
                    </Select>
                    {errors.id_curso && (
                        <p className="text-sm text-destructive">
                            {errors.id_curso}
                        </p>
                    )}
                </div>

                <div className="space-y-2">
                    <Label>Videos de TikTok</Label>
                    <div className="flex gap-2">
                        <Input
                            type="text"
                            value={tikTokInput}
                            onChange={(e) => setTikTokInput(e.target.value)}
                            placeholder="ID del video de TikTok (ej: 7328708521718025474)"
                            className="flex-1"
                            onKeyDown={(e) => {
                                if (e.key === "Enter") {
                                    e.preventDefault();
                                    addTikTokKey();
                                }
                            }}
                        />
                        <Button
                            type="button"
                            onClick={addTikTokKey}
                            disabled={!tikTokInput.trim()}
                        >
                            Agregar
                        </Button>
                    </div>

                    {errors.redes && (
                        <p className="text-sm text-destructive">
                            {errors.redes}
                        </p>
                    )}

                    {/* Lista de IDs agregados */}
                    <div className="mt-2 space-y-1">
                        {data.redes.map((red, index) => (
                            <div key={index} className="flex items-center justify-between p-2 rounded">
                                <span className="text-sm">{red.key}</span>
                                <Button
                                    type="button"
                                    variant="ghost"
                                    size="icon"
                                    className="h-6 w-6"
                                    onClick={() => removeTikTokKey(index)}
                                >
                                    <X className="h-4 w-4 text-destructive" />
                                </Button>
                            </div>
                        ))}
                    </div>


                    {/* Previsualización de videos */}
                    <div className="flex overflow-x-auto pb-4 space-x-4 w-full">
                        {data.redes.map((red, index) => (
                            <TikTokEmbed
                                key={index}
                                red={red}
                                index={index}
                                removeTikTokKey={removeTikTokKey}
                            />
                        ))}
                    </div>
                </div>
                {/* Video preview section */}
                <div className="space-y-2">
                    <Label>Videos</Label>
                    <input
                        type="file"
                        ref={videoInputRef}
                        onChange={(e) => handleVideoChange(e.target.files)}
                        multiple
                        accept="video/*"
                        className="hidden"
                    />
                    <Button
                        type="button"
                        onClick={() => videoInputRef.current?.click()}
                        className="ms-4"
                    >
                        Agregar Videos
                    </Button>
                    <div className="grid grid-cols-1 md:grid-cols-2 gap-4 mt-4">
                        {previewVideos.map((preview, index) => (
                            <div key={index} className="relative group">
                                <video
                                    src={
                                        preview.url &&
                                            preview.url.startsWith("blob:")
                                            ? preview.url
                                            : preview.url
                                                ? preview.url.startsWith("/")
                                                    ? preview.url
                                                    : "/" + preview.url
                                                : ""
                                    }
                                    className="w-full h-auto rounded-lg border"
                                    controls
                                />
                                <Button
                                    type="button"
                                    variant="ghost"
                                    size="icon"
                                    className="absolute top-2 right-2 opacity-0 group-hover:opacity-100 transition-opacity bg-white/80 hover:bg-white"
                                    onClick={() => removeVideo(index)}
                                >
                                    <X className="h-4 w-4 text-destructive" />
                                </Button>
                                <div className="mt-1 text-sm text-muted-foreground truncate">
                                    {preview.name}
                                </div>
                            </div>
                        ))}
                    </div>
                </div>
                {/* Imágenes preview section */}
                <div className="space-y-2">
                    <Label>Imágenes</Label>
                    <input
                        type="file"
                        ref={imagenInputRef}
                        onChange={(e) => handleImagenChange(e.target.files)}
                        multiple
                        accept="image/*"
                        className="hidden"
                    />
                    <Button
                        type="button"
                        className="ms-4"
                        onClick={() => imagenInputRef.current?.click()}
                    >
                        Agregar Imágenes
                    </Button>
                    <div className="grid grid-cols-1 md:grid-cols-3 gap-4 mt-4">
                        {previewImagenes.map((preview, index) => (
                            <div key={index} className="relative group">
                                <img
                                    src={
                                        preview.url &&
                                            preview.url.startsWith("blob:")
                                            ? preview.url
                                            : preview.url
                                                ? preview.url.startsWith("/")
                                                    ? preview.url
                                                    : "/" + preview.url
                                                : ""
                                    }
                                    alt={preview.name}
                                    className="w-full h-auto rounded-lg border"
                                />
                                <Button
                                    type="button"
                                    variant="ghost"
                                    size="icon"
                                    className="absolute top-2 right-2 opacity-0 group-hover:opacity-100 transition-opacity bg-white/80 hover:bg-white"
                                    onClick={() => removeImagen(index)}
                                >
                                    <X className="h-4 w-4 text-destructive" />
                                </Button>
                                <div className="mt-1 text-sm text-muted-foreground truncate">
                                    {preview.name}
                                </div>
                            </div>
                        ))}
                    </div>
                </div>

                {/* PDFs preview section */}
                <div className="space-y-2">
                    <Label>PDFs</Label>
                    <input
                        type="file"
                        ref={pdfInputRef}
                        onChange={(e) => handlePdfChange(e.target.files)}
                        multiple
                        accept="application/pdf"
                        className="hidden"
                    />
                    <Button
                        type="button"
                        className="ms-4"
                        onClick={() => pdfInputRef.current?.click()}
                    >
                        Agregar PDFs
                    </Button>
                    <div className="grid grid-cols-1 gap-4 mt-4">
                        {previewPdfs.map((preview, index) => (
                            <div key={index} className="">
                                <div className="flex items-center justify-between p-3 border rounded-lg">
                                    <div className="flex items-center gap-3">
                                        <FileText className="h-5 w-5 text-primary" />
                                        <span className="text-sm truncate">
                                            {preview.name}
                                        </span>
                                    </div>
                                    <Button
                                        type="button"
                                        variant="ghost"
                                        size="icon"
                                        className="h-8 w-8"
                                        onClick={() => removePdf(index)}
                                    >
                                        <X className="h-4 w-4 text-destructive" />
                                    </Button>
                                </div>

                                <div>
                                    <iframe
                                        src={
                                            preview.url &&
                                                preview.url.startsWith("blob:")
                                                ? preview.url
                                                : preview.url
                                                    ? preview.url.startsWith("/")
                                                        ? preview.url
                                                        : "/" + preview.url
                                                    : ""
                                        }
                                        width="100%"
                                        height="600px"
                                        style={{ border: "none" }}
                                    ></iframe>
                                </div>
                            </div>
                        ))}
                    </div>
                </div>
                {/* Similar corrections for Image and PDF sections */}

                <div className="flex justify-end pt-4">
                    <Button type="submit" disabled={processing}>
                        {processing ? "Guardando..." : "Guardar Materiales"}
                    </Button>
                </div>
            </form>
        </LayoutPrivate>
    );
}
