import { Button } from "@/Components/ui/button";
import { Label } from "@/Components/ui/label";
import { Input } from "@/Components/ui/input";
import { useForm, usePage } from "@inertiajs/react";
import { PageProps } from "@/types/param";
import {
    Select,
    SelectContent,
    SelectItem,
    SelectTrigger,
    SelectValue,
} from "@/Components/ui/select";
import { FileText, X } from "lucide-react";
import { CursoMaterial } from "@/types/curso-material";
import { Curso } from "@/types/curso";
import LayoutPrivate from "@/Layouts/Private/layout";
import { FormEventHandler, useId, useRef, useState } from "react";
import { toast } from "sonner";
import { validateInputText } from "@/Utils/validations";
import { TikTokEmbed } from "@/Components/TikTokEmbed";

interface Redes {
    key: string;
}

export default function Create() {
    const idNombre = useId();
    const idkey = useId();

    const { auth, cursos } = usePage<PageProps<CursoMaterial>>().props;

    const videoInputRef = useRef<HTMLInputElement>(null);
    const imagenInputRef = useRef<HTMLInputElement>(null);
    const pdfInputRef = useRef<HTMLInputElement>(null);

    const { data, setData, post, reset, clearErrors, processing, errors } =
        useForm<{
            nombre: string;
            id_curso: string;
            videos: File[];
            imagenes: File[];
            pdfs: File[];
            redes: Redes[];
        }>({
            nombre: "",
            id_curso: "",
            videos: [],
            imagenes: [],
            pdfs: [],
            redes: [],
        });

    const [previewVideos, setPreviewVideos] = useState<string[]>([]);
    const [previewRedes, setPreviewRedes] = useState<string[]>([]);
    const [tikTokInput, setTikTokInput] = useState("");
    const [previewImagenes, setPreviewImagenes] = useState<string[]>([]);
    const [previewPdfs, setPreviewPdfs] = useState<string[]>([]);

    const handleVideoChange = (files: FileList | null) => {
        if (!files) return;
        const fileArray = Array.from(files);
        const videoFiles = fileArray.filter((file) =>
            file.type.startsWith("video/")
        );

        if (videoFiles.length !== fileArray.length) {
            toast.warning("Algunos archivos no son videos y fueron ignorados");
        }

        setData("videos", [...data.videos, ...videoFiles]);
        const newPreviews = videoFiles.map((file) => URL.createObjectURL(file));
        setPreviewVideos([...previewVideos, ...newPreviews]);
    };

    const handleImagenChange = (files: FileList | null) => {
        if (!files) return;
        const fileArray = Array.from(files);
        const imageFiles = fileArray.filter((file) =>
            file.type.startsWith("image/")
        );

        if (imageFiles.length !== fileArray.length) {
            toast.warning(
                "Algunos archivos no son imágenes y fueron ignorados"
            );
        }

        setData("imagenes", [...data.imagenes, ...imageFiles]);
        const newPreviews = imageFiles.map((file) => URL.createObjectURL(file));
        setPreviewImagenes([...previewImagenes, ...newPreviews]);
    };

    const addTikTokKey = () => {
        if (tikTokInput.trim()) {
            const newRedes = [...data.redes, { key: tikTokInput.trim() }];
            setData("redes", newRedes);
            setPreviewRedes([...previewRedes, tikTokInput.trim()]);
            setTikTokInput("");
        }
    };

    const handlePdfChange = (files: FileList | null) => {
        if (!files) return;
        const fileArray = Array.from(files);
        const pdfFiles = fileArray.filter(
            (file) => file.type === "application/pdf"
        );

        if (pdfFiles.length !== fileArray.length) {
            toast.warning("Algunos archivos no son PDFs y fueron ignorados");
        }

        setData("pdfs", [...data.pdfs, ...pdfFiles]);
        const newPreviews = pdfFiles.map((file) => file.name);
        setPreviewPdfs([...previewPdfs, ...newPreviews]);
    };

    const removeVideo = (index: number) => {
        const newFiles = [...data.videos];
        newFiles.splice(index, 1);
        setData("videos", newFiles);

        const newPreviews = [...previewVideos];
        URL.revokeObjectURL(newPreviews[index]);
        newPreviews.splice(index, 1);
        setPreviewVideos(newPreviews);
    };

    const removeImagen = (index: number) => {
        const newFiles = [...data.imagenes];
        newFiles.splice(index, 1);
        setData("imagenes", newFiles);

        const newPreviews = [...previewImagenes];
        URL.revokeObjectURL(newPreviews[index]);
        newPreviews.splice(index, 1);
        setPreviewImagenes(newPreviews);
    };

    const removePdf = (index: number) => {
        const newFiles = [...data.pdfs];
        newFiles.splice(index, 1);
        setData("pdfs", newFiles);

        const newPreviews = [...previewPdfs];
        newPreviews.splice(index, 1);
        setPreviewPdfs(newPreviews);
    };

    const removeTikTokKey = (index: number) => {
        const newRedes = [...data.redes];
        newRedes.splice(index, 1);
        setData("redes", newRedes);

        const newPreviews = [...previewRedes];
        newPreviews.splice(index, 1);
        setPreviewRedes(newPreviews);
    };

    const triggerVideoInput = () => videoInputRef.current?.click();
    const triggerImagenInput = () => imagenInputRef.current?.click();
    const triggerPdfInput = () => pdfInputRef.current?.click();

    const submit: FormEventHandler = (e) => {
        e.preventDefault();

        const formData = new FormData();
        formData.append("nombre", data.nombre);
        formData.append("id_curso", data.id_curso);

        data.videos.forEach((file, index) => {
            formData.append(`videos[${index}]`, file);
        });

        data.imagenes.forEach((file, index) => {
            formData.append(`imagenes[${index}]`, file);
        });

        data.pdfs.forEach((file, index) => {
            formData.append(`pdfs[${index}]`, file);
        });

        data.redes.forEach((red, index) => {
            formData.append(`redes[${index}][key]`, red.key);
        });

        post(route("cursos-materiales.store"), {
            data: formData,
            preserveScroll: true,
            onSuccess: () => {
                reset();
                toast.success("Materiales creados exitosamente");
                setPreviewRedes([]);
                previewVideos.forEach((url) => URL.revokeObjectURL(url));
                previewImagenes.forEach((url) => URL.revokeObjectURL(url));
                setPreviewVideos([]);
                setPreviewImagenes([]);
                setPreviewPdfs([]);
            },
            onError: (errors) => {
                Object.entries(errors).forEach(([field, message]) => {
                    toast.error(`${field}: ${message}`);
                });
            },
        });
    };

    return (
        <LayoutPrivate user={auth.user} header={"Sesiones de Estudio"}>
            <form
                onSubmit={submit}
                encType="multipart/form-data"
                className="w-full max-w-4xl mx-auto space-y-6"
            >
                {/* Nombre */}
                <div className="space-y-2">
                    <Label htmlFor={idNombre}>Nombre de Sesión</Label>
                    <Input
                        id={idNombre}
                        type="text"
                        value={data.nombre}
                        className="w-full"
                        placeholder="Nombre de Sesión"
                        onChange={(e) => setData("nombre", e.target.value)}
                        required
                    />
                    {errors.nombre && (
                        <p className="text-sm text-destructive">
                            {errors.nombre}
                        </p>
                    )}
                </div>

                {/* Curso */}
                <div className="space-y-2">
                    <Label>Curso</Label>
                    <Select
                        value={data.id_curso}
                        onValueChange={(value) => setData("id_curso", value)}
                    >
                        <SelectTrigger className="w-full">
                            <SelectValue placeholder="Seleccione un curso" />
                        </SelectTrigger>
                        <SelectContent>
                            {cursos.map((curso: Curso) => (
                                <SelectItem
                                    key={curso.id}
                                    value={curso.id.toString()}
                                >
                                    {curso.nombre}
                                </SelectItem>
                            ))}
                        </SelectContent>
                    </Select>
                    {errors.id_curso && (
                        <p className="text-sm text-destructive">
                            {errors.id_curso}
                        </p>
                    )}
                </div>
                {/* ck */}
                <div className="space-y-2">
                    <Label>Videos de TikTok</Label>
                    <div className="flex gap-2">
                        <Input
                            type="text"
                            value={tikTokInput}
                            onChange={(e) => setTikTokInput(e.target.value)}
                            placeholder="ID del video de TikTok (ej: 7328708521718025474)"
                            className="flex-1"
                            onKeyDown={(e) => {
                                if (e.key === "Enter") {
                                    e.preventDefault();
                                    addTikTokKey();
                                }
                            }}
                        />
                        <Button
                            type="button"
                            onClick={addTikTokKey}
                            disabled={!tikTokInput.trim()}
                        >
                            Agregar
                        </Button>
                    </div>

                    {errors.redes && (
                        <p className="text-sm text-destructive">
                            {errors.redes}
                        </p>
                    )}

                    {/* Lista de IDs agregados */}
                    <div className="mt-2 space-y-1">
                        {data.redes.map((red, index) => (
                            <div
                                key={index}
                                className="flex items-center justify-between p-2 rounded"
                            >
                                <span className="text-sm">{red.key}</span>
                                <Button
                                    type="button"
                                    variant="ghost"
                                    size="icon"
                                    className="h-6 w-6"
                                    onClick={() => removeTikTokKey(index)}
                                >
                                    <X className="h-4 w-4 text-destructive" />
                                </Button>
                            </div>
                        ))}
                    </div>

                    {/* Previsualización de videos */}
                    <div className="flex overflow-x-auto pb-4 space-x-4 w-full">
                        {data.redes.map((red, index) => (
                            <TikTokEmbed
                                key={index}
                                red={red}
                                index={index}
                                removeTikTokKey={removeTikTokKey}
                            />
                        ))}
                    </div>
                </div>
                {/* Videos */}
                <div className="space-y-2">
                    <Label>Videos</Label>
                    <input
                        type="file"
                        ref={videoInputRef}
                        onChange={(e) => handleVideoChange(e.target.files)}
                        multiple
                        accept="video/*"
                        className="hidden"
                    />
                    <Button
                        type="button"
                        variant="outline"
                        className="ms-4"
                        onClick={triggerVideoInput}
                    >
                        Agregar Videos
                    </Button>
                    {errors.videos && (
                        <p className="text-sm text-destructive">
                            {errors.videos}
                        </p>
                    )}
                    <div className="grid grid-cols-1 md:grid-cols-2 gap-4 mt-4">
                        {previewVideos.map((preview, index) => (
                            <div key={index} className="relative group">
                                <video
                                    src={preview}
                                    className="w-full h-auto rounded-lg border"
                                    controls
                                />
                                <Button
                                    type="button"
                                    variant="ghost"
                                    size="icon"
                                    className="absolute top-2 right-2 opacity-0 group-hover:opacity-100 transition-opacity bg-white/80 hover:bg-white"
                                    onClick={() => removeVideo(index)}
                                >
                                    <X className="h-4 w-4 text-destructive" />
                                </Button>
                                <div className="mt-1 text-sm text-muted-foreground truncate">
                                    {data.videos[index].name}
                                </div>
                            </div>
                        ))}
                    </div>
                </div>

                {/* Imágenes */}
                <div className="space-y-2">
                    <Label>Imágenes</Label>
                    <input
                        type="file"
                        ref={imagenInputRef}
                        onChange={(e) => handleImagenChange(e.target.files)}
                        multiple
                        accept="image/*"
                        className="hidden"
                    />
                    <Button
                        type="button"
                        variant="outline"
                        className="ms-4"
                        onClick={triggerImagenInput}
                    >
                        Agregar Imágenes
                    </Button>
                    {errors.imagenes && (
                        <p className="text-sm text-destructive">
                            {errors.imagenes}
                        </p>
                    )}
                    <div className="grid grid-cols-1 md:grid-cols-2 lg:grid-cols-3 gap-4 mt-4">
                        {previewImagenes.map((preview, index) => (
                            <div key={index} className="relative group">
                                <img
                                    src={preview}
                                    alt="Previsualización"
                                    className="w-full h-48 object-cover rounded-lg border"
                                />
                                <Button
                                    type="button"
                                    variant="ghost"
                                    size="icon"
                                    className="absolute top-2 right-2 opacity-0 group-hover:opacity-100 transition-opacity bg-white/80 hover:bg-white"
                                    onClick={() => removeImagen(index)}
                                >
                                    <X className="h-4 w-4 text-destructive" />
                                </Button>
                                <div className="mt-1 text-sm text-muted-foreground truncate">
                                    {data.imagenes[index].name}
                                </div>
                            </div>
                        ))}
                    </div>
                </div>

                {/* PDFs */}
                <div className="space-y-2">
                    <Label>Documentos PDF</Label>
                    <input
                        type="file"
                        ref={pdfInputRef}
                        onChange={(e) => handlePdfChange(e.target.files)}
                        multiple
                        accept="application/pdf"
                        className="hidden"
                    />
                    <Button
                        type="button"
                        variant="outline"
                        className="ms-4"
                        onClick={triggerPdfInput}
                    >
                        Agregar PDFs
                    </Button>
                    {errors.pdfs && (
                        <p className="text-sm text-destructive">
                            {errors.pdfs}
                        </p>
                    )}
                    <div className="space-y-2 mt-4">
                        {previewPdfs.map((name, index) => (
                            <div
                                key={index}
                                className="flex items-center justify-between p-3 border rounded-lg"
                            >
                                <div className="flex items-center space-x-2">
                                    <FileText className="h-5 w-5 text-primary" />
                                    <span className="text-sm truncate">
                                        {name}
                                    </span>
                                </div>
                                <Button
                                    type="button"
                                    variant="ghost"
                                    size="icon"
                                    onClick={() => removePdf(index)}
                                >
                                    <X className="h-4 w-4 text-destructive" />
                                </Button>
                            </div>
                        ))}
                    </div>
                </div>

                {/* Botón de enviar */}
                <div className="flex justify-end pt-4">
                    <Button type="submit" disabled={processing}>
                        {processing ? "Guardando..." : "Guardar Materiales"}
                    </Button>
                </div>
            </form>
        </LayoutPrivate>
    );
}
