import {
    AlertDialog,
    AlertDialogContent,
    AlertDialogDescription,
    AlertDialogHeader,
    AlertDialogTitle,
} from "@/Components/ui/alert-dialog";
import { Button } from "@/Components/ui/button";
import { Input } from "@/Components/ui/input";
import { Label } from "@/Components/ui/label";
import { useForm } from "@inertiajs/react";
import { FormEventHandler, useId } from "react";
import { useCursoStore } from "@/Store/useCursoStore";
import { toast } from "sonner";
import { Page } from "@inertiajs/core";
import { validateInputText } from "@/Utils/validations";

export function ModalCreateOrEdit() {
    const idNombre = useId();
    const { curso, openModalCreateOrEditCurso, setOpenModalCreateOrEditCurso } =
        useCursoStore();

    const { data, setData, post, put, reset, clearErrors, processing, errors } =
        useForm<{
            id: number | null;
            nombre: string;
        }>({
            id: curso?.id ?? null,
            nombre: curso?.nombre || "",
        });

    const submit: FormEventHandler = (e) => {
        e.preventDefault();

        const options = {
            preserveScroll: true,
            onSuccess: (response: Page<{ status?: { success?: string } }>) => {
                reset();
                const successMessage =
                    response.props.status?.success ||
                    "Operación realizada con éxito.";
                toast.success(successMessage);
                setOpenModalCreateOrEditCurso(false);
            },
            onError: (response: { nombre?: string }) => {
                toast.error(response.nombre || "Error en el formulario.");
            },
        };

        if (data.id) {
            put(route("cursos.update", { curso: data.id }), options);
        } else {
            post(route("cursos.store"), options);
        }
        clearErrors();
    };

    const handleNameChange = (e: React.ChangeEvent<HTMLInputElement>) => {
        const value = e.target.value;
        if (validateInputText(value)) {
            setData("nombre", value); 
        }
    };

    return (
        <AlertDialog
            open={openModalCreateOrEditCurso}
            onOpenChange={() => setOpenModalCreateOrEditCurso(false)}
        >
            <AlertDialogContent>
                <AlertDialogHeader>
                    <AlertDialogTitle>
                        {curso?.id ? "Editar" : "Crear"} Curso
                    </AlertDialogTitle>
                    <AlertDialogDescription>
                        {curso?.id
                            ? "Modifica los datos del curso y guarda los cambios."
                            : "Ingresa los datos para registrar un nuevo curso."}
                    </AlertDialogDescription>
                </AlertDialogHeader>
                <form onSubmit={submit}>
                    <Label htmlFor={idNombre}>Nombre</Label>
                    <Input
                        id={idNombre}
                        type="text"
                        value={data.nombre}
                        className="mt-1 block w-full"
                        placeholder="Nombre del curso"
                        onChange={handleNameChange}
                        required
                    />
                    {errors.nombre && (
                        <Label className="text-destructive" htmlFor={idNombre}>
                            {errors.nombre}
                        </Label>
                    )}
                    <footer className="flex justify-around gap-4 mt-4">
                        <Button
                            type="button"
                            onClick={() => setOpenModalCreateOrEditCurso(false)}
                            variant="secondary"
                        >
                            Cancelar
                        </Button>
                        <Button type="submit" disabled={processing}>
                            Guardar
                        </Button>
                    </footer>
                </form>
            </AlertDialogContent>
        </AlertDialog>
    );
}
