<?php

namespace App\Models;

use Illuminate\Database\Eloquent\Factories\HasFactory;
use Illuminate\Database\Eloquent\Model;
use Laravel\Sanctum\HasApiTokens;
use Illuminate\Foundation\Auth\User as Authenticatable;

/**
 * Modelo Estudiante
 * 
 * Representa un estudiante dentro del sistema.
 *
 * @package App\Models
 */
class Estudiante extends Authenticatable
{
    use HasApiTokens, HasFactory;

    /**
     * Nombre de la tabla en la base de datos.
     * 
     * @var string
     */
    protected $table = 'estudiantes';

    /**
     * Atributos que se pueden asignar masivamente.
     * 
     * @var array<int, string>
     */
    protected $fillable = [
        'dni',
        'nombres',
        'apellido_paterno',
        'apellido_materno',
        'celular',
        'celular_apoderado',
        'password',
    ];

    /**
     * Atributos que deben estar ocultos al serializar el modelo.
     * 
     * @var array<int, string>
     */
    protected $hidden = [
        'password',
    ];

    /**
     * Conversión de atributos a tipos específicos.
     * 
     * @var array<string, string>
     */
    protected $casts = [
        'password' => 'hashed',
    ];

    /**
     * Relación con la tabla intermedia `cursos_estudiantes`.
     * 
     * Un estudiante puede estar inscrito en varios cursos.
     *
     * @return \Illuminate\Database\Eloquent\Relations\HasMany
     */
    public function cursos_estudiantes()
    {
        return $this->hasMany(CursoEstudiante::class, 'id_estudiante');
    }

    /**
     * Relación con el modelo Curso.
     * 
     * Un estudiante puede estar inscrito en varios cursos.
     *
     * @return \Illuminate\Database\Eloquent\Relations\BelongsToMany
     */
    public function cursos()
    {
        return $this->belongsToMany(Curso::class, 'cursos_estudiantes', 'id_estudiante', 'id_curso');
    }

     /**
     * Relación con el modelo Curso.
     * 
     * Un estudiante puede estar inscrito en varios grados.
     *
     * @return \Illuminate\Database\Eloquent\Relations\BelongsToMany
     */
    public function grado()
    {
        return $this->belongsToMany(Grado::class, 'grados_estudiantes', 'id_estudiante', 'id_grado');
    }

    /**
     * Relación con el modelo Nota.
     * 
     * Un estudiante puede estar inscrito en varios notas.
     *
     * @return \Illuminate\Database\Eloquent\Relations\hasMany
     */
    public function notas()
    {
        return $this->hasMany(Nota::class,'id_estudiante');
    }

    /**
     * Relación con el modelo Asistencia.
     * 
     * Un estudiante puede estar inscrito en varios asistencias.
     *
     * @return \Illuminate\Database\Eloquent\Relations\hasMany
     */
    public function asistencias()
    {
        return $this->hasMany(Asistencia::class, 'id_estudiante');
    }

    /**
     * Relación con el modelo Conducta.
     * 
     * Un estudiante puede estar inscrito en varios conductas.
     *
     * @return \Illuminate\Database\Eloquent\Relations\hasMany
     */
    public function conductas()
    {
        return $this->hasMany(Conducta::class, 'id_estudiante');
    }
    
}
