<?php

namespace App\Http\Requests;

use Illuminate\Foundation\Http\FormRequest;

class CursoMaterialEditRequest extends FormRequest
{
    /**
     * Determine if the user is authorized to make this request.
     */
    public function authorize(): bool
    {
        return true;
    }

    /**
     * Get the validation rules that apply to the request.
     *
     * @return array<string, \Illuminate\Contracts\Validation\ValidationRule|array<mixed>|string>
     */
    public function rules(): array
    {
        return [
            'nombre' => 'required|string|max:255',
            'id_curso' => 'required|exists:cursos,id',
            
            // Archivos nuevos
            'videos' => 'nullable|array',
            'videos.*' => 'file|mimetypes:video/mp4,video/avi,video/mpeg,video/quicktime,video/x-msvideo,video/x-matroska|max:512000', // 500MB
            'imagenes' => 'nullable|array',
            'imagenes.*' => 'file|image|mimes:jpeg,png,jpg,gif,webp|max:51200', // 50MB
            'pdfs' => 'nullable|array',
            'pdfs.*' => 'file|mimes:pdf|max:51200', // 50MB

            'redes.*.key' => 'required|string|max:50',
            
            // IDs para eliminar
            'deleted_video_ids' => 'sometimes|array',
            'deleted_video_ids.*' => 'integer|exists:cursos_materiales_videos,id',
            'deleted_imagen_ids' => 'sometimes|array',
            'deleted_imagen_ids.*' => 'integer|exists:cursos_materiales_imagenes,id',
            'deleted_pdf_ids' => 'sometimes|array',
            'deleted_pdf_ids.*' => 'integer|exists:cursos_materiales_pdfs,id',
            
            // Validación de límite total (opcional)
            'videos' => [
                'nullable',
                'array',
                function ($attribute, $value, $fail) {
                    $totalSize = array_reduce($value, function ($sum, $file) {
                        return $sum + $file->getSize();
                    }, 0);
                    
                    if ($totalSize > 1024 * 1024 * 1024) { // 1GB total
                        $fail('El tamaño total de los videos no puede exceder 1GB.');
                    }
                },
            ],
        ];
    }
    
    public function messages(): array
    {
        return [
            // Mensajes para campos básicos
            'nombre.required' => 'El nombre del material es obligatorio.',
            'nombre.string' => 'El nombre debe ser una cadena de texto válida.',
            'nombre.max' => 'El nombre no debe exceder los 255 caracteres.',

            'id_curso.required' => 'Debe seleccionar un curso válido.',
            'id_curso.exists' => 'El curso seleccionado no existe en el sistema.',

            // Mensajes para videos
            'videos.array' => 'Los videos deben enviarse como arreglo.',
            'videos.*.file' => 'Cada video debe ser un archivo válido.',
            'videos.*.mimetypes' => 'Formatos permitidos: MP4, AVI, MPEG, MOV, MKV.',
            'videos.*.max' => 'Cada video no debe exceder 500MB.',

            'redes.array' => 'El formato de redes sociales no es válido.',
            'redes.*.key.required' => 'El identificador de la red social es obligatorio.',
            'redes.*.key.string' => 'El identificador debe ser texto.',
            'redes.*.key.max' => 'El identificador no debe exceder 50 caracteres.',
            'redes.*.key.regex' => 'El formato del identificador no es válido para esta red social.',
           
            // Mensajes para imágenes
            'imagenes.array' => 'Las imágenes deben enviarse como arreglo.',
            'imagenes.*.file' => 'Cada imagen debe ser un archivo válido.',
            'imagenes.*.image' => 'Solo se permiten archivos de imagen.',
            'imagenes.*.mimes' => 'Formatos permitidos: JPEG, PNG, JPG, GIF, WEBP.',
            'imagenes.*.max' => 'Cada imagen no debe exceder 50MB.',

            // Mensajes para PDFs
            'pdfs.array' => 'Los PDFs deben enviarse como arreglo.',
            'pdfs.*.file' => 'Cada PDF debe ser un archivo válido.',
            'pdfs.*.mimes' => 'Solo se permiten archivos PDF.',
            'pdfs.*.max' => 'Cada PDF no debe exceder 50MB.',

            // Mensajes para eliminación
            'deleted_video_ids.array' => 'Los IDs de videos a eliminar deben ser un arreglo.',
            'deleted_video_ids.*.integer' => 'Cada ID de video debe ser un número entero.',
            'deleted_video_ids.*.exists' => 'Uno o más IDs de videos no existen.',

            'deleted_imagen_ids.array' => 'Los IDs de imágenes a eliminar deben ser un arreglo.',
            'deleted_imagen_ids.*.integer' => 'Cada ID de imagen debe ser un número entero.',
            'deleted_imagen_ids.*.exists' => 'Uno o más IDs de imágenes no existen.',

            'deleted_pdf_ids.array' => 'Los IDs de PDFs a eliminar deben ser un arreglo.',
            'deleted_pdf_ids.*.integer' => 'Cada ID de PDF debe ser un número entero.',
            'deleted_pdf_ids.*.exists' => 'Uno o más IDs de PDFs no existen.',
        ];
    }

    // /**
    //  * Prepare the data for validation.
    //  */
    // protected function prepareForValidation()
    // {
    //     // Convertir arrays vacíos a null para evitar errores de validación
    //     $this->merge([
    //         'videos' => $this->videos ?: null,
    //         'imagenes' => $this->imagenes ?: null,
    //         'pdfs' => $this->pdfs ?: null,
    //         'deleted_video_ids' => $this->deleted_video_ids ?: null,
    //         'deleted_imagen_ids' => $this->deleted_imagen_ids ?: null,
    //         'deleted_pdf_ids' => $this->deleted_pdf_ids ?: null,
    //     ]);
    // }
}