<?php

namespace App\Http\Controllers;

use App\Http\Requests\UsuarioRequest;
use App\Models\User;
use Illuminate\Support\Facades\Hash;
use Inertia\Inertia;
use Inertia\Response;

/**
 * Controlador UsuarioController
 *
 * Maneja la gestión de usuarios con Inertia.js.
 */
class UsuarioController extends Controller
{
    /**
     * Muestra la lista de usuarios.
     *
     * @return Response
     */
    public function index(): Response
    {
        $usuarios = User::latest()->get();

        return Inertia::render('Usuarios/Index', [
            'usuarios' => $usuarios,
        ]);
    }

    /**
     * Guarda un nuevo usuario.
     *
     * @param UsuarioRequest $request
     * @return \Illuminate\Http\RedirectResponse
     */
    public function store(UsuarioRequest $request)
    {
        $data = $request->validated();
        $data['password'] = Hash::make($data['password']);

        User::create($data);

        return redirect()->back()->with('success', 'Usuario creado correctamente.');
    }

    /**
     * Actualiza la información de un usuario.
     *
     * @param UsuarioRequest $request
     * @param User $usuario
     * @return \Illuminate\Http\RedirectResponse
     */
    public function update(UsuarioRequest $request, User $usuario)
    {
        $data = $request->validated();

        if ($request->filled('password')) {
            $data['password'] = Hash::make($data['password']);
        } else {
            unset($data['password']); // No actualizar la contraseña si no se envía
        }

        $usuario->update($data);

        return redirect()->back()->with('success', 'Usuario actualizado correctamente.');
    }

    /**
     * Elimina un usuario.
     *
     * @param User $usuario
     * @return \Illuminate\Http\RedirectResponse
     */
    public function destroy(User $usuario)
    {
        $usuario->delete();

        return redirect()->back()->with('success', 'Usuario eliminado correctamente.');
    }
}
