<?php

namespace App\Http\Controllers;

use App\Http\Requests\NotaRequest;
use App\Models\Nota;
use Inertia\Inertia;
use Inertia\Response;
use Illuminate\Support\Facades\Storage;

/**
 * Controlador NotaController
 *
 * Maneja la gestión de notas usando Inertia.js.
 */
class NotaController extends Controller
{
    /**
     * Muestra la lista de notas.
     *
     * @return Response
     */
    public function index()
    {
        $perPage = request('per_page', 10);
        $search = request('search', '');

        $notas = Nota::latest()
            ->where('fecha', 'like', "%{$search}%")
            ->paginate($perPage)
            ->appends(['search' => $search, 'per_page' => $perPage]);

        return Inertia::render('Notas/Index', [
            'notas' => $notas,
            'filters' => request()->only(['search', 'per_page']),
            'status' => session()->only(['success', 'error']),
        ]);
    }

   /**
     * Almacena una nueva nota en la base de datos.
     *
     * @param NotaRequest $request
     * @return \Illuminate\Http\RedirectResponse
     */
    public function store(NotaRequest $request)
    {
        $nota = Nota::create($request->validated());
        return redirect()->back()->with('success', 'Nota: ' . $nota->fecha . ' creado exitosamente.');
    }

    /**
     * Actualiza una nota existente en la base de datos.
     *
     * @param NotaRequest $request
     * @param int $id
     * @return \Illuminate\Http\RedirectResponse
     */
    public function update(NotaRequest $request, int $id)
    {
        $nota = Nota::findOrFail($id);
    
        $nota->update($request->validated());

        return redirect()->back()->with('success', 'Nota: ' . $nota->fecha . ' actualizado exitosamente.');
    
    }
    
    /**
     * Elimina una nota de la base de datos junto con su imagen si existe.
     *
     * @param int $id_nota
     * @return \Illuminate\Http\RedirectResponse
     */
    public function destroy(int $id)
    {
       $nota = Nota::findOrFail($id);
        if (!$nota) {
            return redirect()->back()->with('error', 'Nota no encontrada.');
        }
        if ($nota->url_imagen) {
            Storage::delete(str_replace('/storage/', 'public/', $nota->url_imagen));
        }
        $nota->delete();
        return redirect()->back()->with('success', 'Nota eliminada exitosamente.');
    }

}
