<?php

namespace App\Http\Controllers;

use App\Http\Requests\GradoEstudianteRequest;
use App\Models\Grado;
use App\Models\GradoEstudiante;
use App\Models\Estudiante;
use Inertia\Inertia;
use Inertia\Response;


/**
 * Controlador GradoEstudianteController
 *
 * Maneja la asignación de estudiantes a grados usando Inertia.js.
 */
class GradoEstudianteController extends Controller
{
    /**
     * Muestra la lista de asignaciones de estudiantes a grados.
     *
     * @return Response
     */
    public function index(): Response
    {

        $gradosEstudiantes = GradoEstudiante::with(['estudiante', 'grado'])->latest()->get();

        return Inertia::render('GradosEstudiantes/Index', [
            'gradosEstudiantes' => $gradosEstudiantes,
            'estudiantes' => Estudiante::all(),
            'grados' => Grado::all(),
        ]);
    }

    /**
     * Asigna un estudiante a un grado.
     *
     * @param GradoEstudianteRequest $request
     * @return \Illuminate\Http\RedirectResponse
     */
    public function store(GradoEstudianteRequest $request)
    {
        GradoEstudiante::create($request->validated());

        return redirect()->back()->with('success', 'Estudiante asignado al grado exitosamente.');
    }

    /**
     * Actualiza la asignación de un estudiante a un grado.
     *
     * @param GradoEstudianteRequest $request
     * @param GradoEstudiante $gradoEstudiante
     * @return \Illuminate\Http\RedirectResponse
     */
    public function update(GradoEstudianteRequest $request, GradoEstudiante $gradoEstudiante)
    {
        $gradoEstudiante->update($request->validated());

        return redirect()->back()->with('success', 'Asignación actualizada exitosamente.');
    }

    /**
     * Elimina una asignación de estudiante a grado.
     *
     * @param GradoEstudiante $gradoEstudiante
     * @return \Illuminate\Http\RedirectResponse
     */
    public function destroy(GradoEstudiante $gradoEstudiante)
    {
        $gradoEstudiante->delete();

        return redirect()->back()->with('success', 'Asignación eliminada exitosamente.');
    }
}
