<?php

namespace App\Http\Controllers;

use App\Http\Requests\GradoRequest;
use App\Models\Grado;
use Inertia\Inertia;
use Inertia\Response;

/**
 * Controlador GradoController
 *
 * Maneja la gestión de grados usando Inertia.js.
 */
class GradoController extends Controller
{
    /**
     * Muestra la lista de grados.
     *
     * @return Response
     */
    public function index()
    {
        $perPage = request('per_page', 10);
        $search = request('search', '');

        $grados = Grado::latest()
            ->where('nombre', 'like', "%{$search}%")
            ->paginate($perPage)
            ->appends(['search' => $search, 'per_page' => $perPage]);

        return Inertia::render('Grados/Index', [
            'grados' => $grados,
            'filters' => request()->only(['search', 'per_page']),
            'status' => session()->only(['success', 'error']),
        ]);
    }

    /**
     * Almacena un nuevo grado en la base de datos.
     *
     * @param GradoRequest $request
     * @return \Illuminate\Http\RedirectResponse
     */
    public function store(GradoRequest $request)
    {
        $grado = Grado::create($request->validated());
        return redirect()->back()->with('success', 'Grado: ' . $grado->nombre . ' creado exitosamente.');
    }

    /**
     * Actualiza un grado existente en la base de datos.
     *
     * @param GradoRequest $request
     * @param Grado $grado
     * @return \Illuminate\Http\RedirectResponse
     */
    public function update(GradoRequest $request, Grado $grado)
    {
        $grado->update($request->validated());

        return redirect()->back()->with('success', 'Grado: ' . $grado->nombre . ' actualizado exitosamente.');
    }

    /**
     * Elimina un grado de la base de datos.
     *
     * @param Grado $grado
     * @return \Illuminate\Http\RedirectResponse
     */
    public function destroy(Grado $grado)
    {
        $grado->delete();

        return redirect()->back()->with('success', 'Grado eliminado exitosamente.');
    }
}
