<?php

    namespace App\Http\Controllers;

    use App\Http\Requests\EstudianteRequest;
    use App\Models\Estudiante;
    use App\Models\Curso;
    use App\Models\Grado;
    use Inertia\Inertia;
    use Inertia\Response;
    use Illuminate\Support\Facades\Hash;

    /**
     * Controlador EstudianteController
     *
     * Maneja la gestión de estudiantes usando Inertia.js.
     */
    class EstudianteController extends Controller
    {
        /**
         * Muestra la lista de estudiantes.
         *
         * @return Response
         */
        public function index(): Response
        {

            $perPage = request('per_page', 10);
            $search = request('search', '');

            $estudiantes = Estudiante::with('cursos', 'grado')->latest()
                ->where('dni', 'like', "%{$search}%")
                ->orWhere('nombres', 'like', "%{$search}%")
                ->orWhere('apellido_paterno', 'like', "%{$search}%")
                ->orWhere('apellido_materno', 'like', "%{$search}%")
                ->orWhere('celular', 'like', "%{$search}%")
                ->orWhere('celular_apoderado', 'like', "%{$search}%")
                ->paginate($perPage)
                ->appends(['search' => $search, 'per_page' => $perPage]);
            $cursos = Curso::all();
            $grados = Grado::all();

            return Inertia::render('Estudiantes/Admin/Index', [
                'estudiantes' => $estudiantes,
                'cursos' => $cursos,
                'grados' => $grados,
                'filters' => request()->only(['search', 'per_page']),
                'status' => session()->only(['success', 'error']),
            ]);
        }

        /**
         * Almacena un nuevo estudiante en la base de datos.
         *
         * @param EstudianteRequest $request
         * @return \Illuminate\Http\RedirectResponse
         */
        public function store(EstudianteRequest $request)
        {
            $requestValidate = $request->validated();

            $requestValidate['password'] = Hash::make($request->dni);

            $estudiante = Estudiante::create($requestValidate);

            $estudiante->cursos()->sync(collect($request->input('cursos'))->pluck('id')->toArray());
            
            $estudiante->grado()->sync(collect($request->input('grado'))->pluck('id')->toArray());

            return redirect()->back()->with('success', 'Estudiante creado exitosamente.');
        }

        /**
         * Actualiza un estudiante existente en la base de datos.
         *
         * @param EstudianteRequest $request
         * @param Estudiante $estudiante
         * @return \Illuminate\Http\RedirectResponse
         */
        public function update(EstudianteRequest $request, Estudiante $estudiante)
        {
            $requestValidate = $request->validated();

            if ($request->has('password') && !empty($request->password)) {
                $requestValidate['password'] = Hash::make($request->password);
            } else {
                unset($requestValidate['password']);
            }

            $estudiante->update($requestValidate);

            $estudiante->cursos()->sync(
                collect($request->input('cursos'))->pluck('id')->toArray()
            );

            $estudiante->grado()->sync(
                collect($request->input('grado'))->pluck('id')->toArray()
            );
            

            return redirect()->back()->with('success', 'Estudiante actualizado exitosamente.');
        }

        /**
         * Elimina un estudiante de la base de datos (Soft Delete).
         *
         * @param Estudiante $estudiante
         * @return \Illuminate\Http\RedirectResponse
         */
        public function destroy(Estudiante $estudiante)
        {
        
            $estudiante->cursos()->detach();
            $estudiante->delete();

            return redirect()->back()->with('success', 'Estudiante eliminado exitosamente.');
        }
    }
