<?php

namespace App\Http\Controllers;

use App\Http\Requests\CursoMaterialEditRequest;
use App\Http\Requests\CursoMaterialRequest;
use App\Models\Curso;
use App\Models\CursoMaterial;
use App\Models\CursoMaterialImagen;
use App\Models\CursoMaterialPdf;
use App\Models\CursoMaterialVideo;
use Illuminate\Support\Str;
use Inertia\Inertia;
use Illuminate\Support\Facades\Log;
use Inertia\Response;

/**
 * Controlador CursoMaterialController
 *
 * Maneja la subida y gestión de materiales de cursos usando Inertia.js.
 */
class CursoMaterialController extends Controller
{
    /**
     * Constructor del controlador
     *
     * Configura límites aumentados para entornos de desarrollo local
     */
    public function __construct()
    {
        if (app()->environment('local')) {
            // Use ini_set() directly and verify changes
            ini_set('memory_limit', '1024M');
            ini_set('post_max_size', '550M');
            ini_set('upload_max_filesize', '500M');
            ini_set('max_execution_time', '1800');
            ini_set('max_input_time', '1800');

            // Verify settings actually applied
            Log::info('PHP Upload Settings', [
                'memory_limit' => ini_get('memory_limit'),
                'upload_max_filesize' => ini_get('upload_max_filesize'),
                'post_max_size' => ini_get('post_max_size'),
            ]);

            $currentUploadSize = ini_get('upload_max_filesize');
            if ((int) $currentUploadSize < 100) {
                // Check if in MB
                Log::error('PHP settings not applied correctly', [
                    'upload_max_filesize' => $currentUploadSize,
                    'post_max_size' => ini_get('post_max_size'),
                ]);
            }
        }
    }

    /**
     * Muestra la lista de materiales de los cursos.
     *
     * @return Response
     */
    public function index(): Response
    {
        $perPage = request('per_page', 10);

        $cursos = Curso::all();

        $cursosMateriales = CursoMaterial::with('cursos_materiales_videos', 'cursos_materiales_imagenes', 'cursos_materiales_pdfs')
            ->latest()
            ->paginate($perPage)
            ->appends(['per_page' => $perPage]);

        return Inertia::render('CursosMateriales/Index', [
            'cursos_materiales' => $cursosMateriales,
            'cursos' => $cursos,
            'filters' => request()->only(['per_page']),
            'status' => session()->only(['success', 'error']),
        ]);
    }

    /**
     * Muestra la vista de materiales a crear.
     *
     * @return Response
     */
    public function create(): Response
    {
        $cursos = Curso::all();
        return Inertia::render('CursosMateriales/Create', [
            'cursos' => $cursos,
        ]);
    }

    /**
     * Guarda un nuevo material para un curso.
     *
     * @param CursoMaterialRequest $request
     * @return \Illuminate\Http\RedirectResponse
     */
    public function store(CursoMaterialRequest $request)
    {
        try {
            // Validar que el curso exista
            $curso = Curso::findOrFail($request->id_curso);

            // Crear el material principal
            $material = CursoMaterial::create([
                'nombre' => $request->nombre,
                'id_curso' => $curso->id,
            ]);

            // Procesar archivos con manejo individual de errores
            $this->handleFiles($material, $request);

            $material->save();

            if ($request->has('redes') && is_array($request->redes)) {
                foreach ($request->redes as $red) {
                    $material->cursos_materiales_redes()->create([
                        'key' => $red['key'],
                        'id_curso_material' => $material->id,
                    ]);
                }
            }
            return redirect()->route('cursos-materiales.index')->with('success', 'Material creado exitosamente');
        } catch (\Exception $e) {
            Log::error('Error al crear material: ' . $e->getMessage(), [
                'exception' => $e,
                'request' => $request->all(),
            ]);

            // Limpiar en caso de error
            if (isset($material) && $material instanceof CursoMaterial) {
                $this->cleanUploads($material);
            }

            return redirect()
                ->back()
                ->with('error', 'Error al guardar el material: ' . $e->getMessage())
                ->withInput();
        }
    }

    /**
     * Maneja todos los tipos de archivos
     */
    protected function handleFiles(CursoMaterial $material, CursoMaterialRequest $request): void
    {
        try {
            // Procesar PDFs
            if ($request->hasFile('pdfs')) {
                foreach ($request->file('pdfs') as $pdf) {
                    $this->processSingleFile($material, $pdf, 'pdfs', 'url_pdf');
                }
            }

            // Procesar Videos
            if ($request->hasFile('videos')) {
                foreach ($request->file('videos') as $video) {
                    $this->processSingleFile($material, $video, 'videos', 'url_video');
                }
            }

            // Procesar Imágenes
            if ($request->hasFile('imagenes')) {
                foreach ($request->file('imagenes') as $imagen) {
                    $this->processSingleFile($material, $imagen, 'imagenes', 'url_imagen');
                }
            }
        } catch (\Exception $e) {
            Log::error('Error al procesar archivos: ' . $e->getMessage());
            throw $e; // Relanzar para manejo global
        }
    }

    /**
     * Procesa un archivo individual
     */
    protected function processSingleFile(CursoMaterial $material, $file, string $type, string $urlField): void
    {
        if (!$file->isValid()) {
            throw new \RuntimeException('Archivo inválido o corrupto: ' . $file->getClientOriginalName());
        }

        if (!file_exists($file->getRealPath())) {
            throw new \RuntimeException('Archivo temporal no encontrado: ' . $file->getClientOriginalName());
        }

        if ($file->getSize() > 524288000) {
            throw new \RuntimeException('El archivo excede el tamaño máximo de 500MB');
        }

        $originalName = $file->getClientOriginalName();
        $size = $file->getSize();
        $mimeType = $file->getMimeType();
        $fileName = $this->generateFileName($file);
        $folder = 'materiales/' . $type;
        $path = $this->saveToPublic($file, $folder, $fileName);

        $relationMethod = 'cursos_materiales_' . $type;

        $material->$relationMethod()->create([
            $urlField => $path,
            'id_curso_material' => $material->id,
            'original_name' => $originalName,
            'size' => $size,
            'mime_type' => $mimeType,
        ]);
    }

    /**
     * Genera un nombre de archivo único
     */
    protected function generateFileName($file): string
    {
        return now()->format('Ymd_His') . '_' . Str::random(8) . '_' . Str::slug(pathinfo($file->getClientOriginalName(), PATHINFO_FILENAME)) . '.' . $file->getClientOriginalExtension();
    }

    /**
     * Guarda el archivo en el directorio public
     */
    protected function saveToPublic($file, string $folder, string $fileName): string
    {
        $publicPath = public_path($folder);

        if (!file_exists($publicPath)) {
            mkdir($publicPath, 0755, true);
        }

        $fullPath = $folder . '/' . $fileName;
        $file->move($publicPath, $fileName);

        return $fullPath;
    }

    /**
     * Limpia archivos subidos en caso de error
     */
    protected function cleanUploads(CursoMaterial $material): void
    {
        try {
            // Eliminar archivos asociados
            $this->deleteRelationFiles($material->cursos_materiales_pdfs, 'url_pdf');
            $this->deleteRelationFiles($material->cursos_materiales_videos, 'url_video');
            $this->deleteRelationFiles($material->cursos_materiales_imagenes, 'url_imagen');

            // Eliminar el material principal
            $material->delete();
        } catch (\Exception $e) {
            Log::error('Error al limpiar uploads: ' . $e->getMessage());
        }
    }

    /**
     * Elimina archivos de una relación
     */
    protected function deleteRelationFiles($relation, string $pathField): void
    {
        if ($relation && $relation->count() > 0) {
            foreach ($relation as $item) {
                try {
                    $filePath = public_path($item->$pathField);
                    if (file_exists($filePath)) {
                        unlink($filePath);
                    }
                    $item->delete();
                } catch (\Exception $e) {
                    Log::error("Error al eliminar archivo {$item->$pathField}: " . $e->getMessage());
                }
            }
        }
    }

    /**
     * Actualiza el material de un curso existente.
     *
     * @param CursoMaterialEditRequest $request
     * @param int $id ID del material del curso
     * @return \Illuminate\Http\RedirectResponse
     */
    public function update(CursoMaterialEditRequest $request, $id)
    {
        // Start transaction FIRST
        \Illuminate\Support\Facades\DB::beginTransaction();

        try {
            $cursoMaterial = CursoMaterial::with(['cursos_materiales_videos', 'cursos_materiales_imagenes', 'cursos_materiales_pdfs'])->findOrFail($id);

            // Process NEW files immediately - most critical operation first
            $this->processNewFilesImmediately($request, $cursoMaterial);

            // Then handle deletions
            $this->handleDeletedFiles($request, $cursoMaterial);

            // Finally update basic data
            $cursoMaterial->update([
                'nombre' => $request->nombre,
                'id_curso' => $request->id_curso,
            ]);

            \Illuminate\Support\Facades\DB::commit();

            $cursoMaterial->save();

            $this->syncRedesSociales($cursoMaterial, $request->redes ?? []);

            return redirect()->route('cursos-materiales.index')->with('success', 'Material actualizado correctamente.');
        } catch (\Exception $e) {
            \Illuminate\Support\Facades\DB::rollBack();

            Log::error('Error al actualizar material: ' . $e->getMessage(), [
                'exception' => $e,
                'material_id' => $id,
                // Don't log entire request as it may contain files
            ]);

            return redirect()
                ->route('cursos-materiales.index')
                ->with('error', 'Error al actualizar el material: ' . $e->getMessage());
        }
    }
    /**
     * Sincroniza las redes sociales del material
     *
     * @param CursoMaterial $cursoMaterial
     * @param array $redesData
     */
    protected function syncRedesSociales(CursoMaterial $cursoMaterial, array $redesData)
    {
        // Obtener IDs existentes del request
        $existingIds = collect($redesData)->filter(fn($red) => isset($red['id']))->pluck('id')->toArray();

        // Eliminar redes que no están en el request
        $cursoMaterial->cursos_materiales_redes()->whereNotIn('id', $existingIds)->delete();

        // Actualizar/Crear redes
        foreach ($redesData as $red) {
            if (isset($red['id'])) {
                $updated = $cursoMaterial
                    ->cursos_materiales_redes()
                    ->where('id', $red['id'])
                    ->update([
                        'key' => $red['key'],
                    ]);

                if ($updated === 0) {
                    // Si no se encontró por ID, lo creamos
                    $cursoMaterial->cursos_materiales_redes()->create([
                        'key' => $red['key'],
                        'id_curso_material' => $cursoMaterial->id,
                    ]);
                }
            } else {
                // Crear nueva red si no tiene ID
                $cursoMaterial->cursos_materiales_redes()->create([
                    'key' => $red['key'],
                    'id_curso_material' => $cursoMaterial->id,
                ]);
            }
        }
    }

    protected function processNewFilesImmediately(CursoMaterialEditRequest $request, CursoMaterial $cursoMaterial): void
    {
        try {
            // Process each file type with immediate storage
            if ($request->hasFile('pdfs')) {
                foreach ($request->file('pdfs') as $pdf) {
                    $this->storeFileImmediately($cursoMaterial, $pdf, 'pdfs', 'url_pdf');
                }
            }

            if ($request->hasFile('videos')) {
                foreach ($request->file('videos') as $video) {
                    $this->storeFileImmediately($cursoMaterial, $video, 'videos', 'url_video');
                }
            }

            if ($request->hasFile('imagenes')) {
                foreach ($request->file('imagenes') as $imagen) {
                    $this->storeFileImmediately($cursoMaterial, $imagen, 'imagenes', 'url_imagen');
                }
            }
        } catch (\Exception $e) {
            Log::error('Error processing new files: ' . $e->getMessage());
            throw $e;
        }
    }

    protected function storeFileImmediately(CursoMaterial $cursoMaterial, $file, string $type, string $urlField): void
    {
        if (!$file->isValid()) {
            throw new \RuntimeException('Invalid file: ' . $file->getClientOriginalName());
        }

        // Get file info BEFORE moving it
        $originalName = $file->getClientOriginalName();
        $mimeType = $file->getMimeType();
        $fileName = $this->generateFileName($file);
        $folder = 'materiales/' . $type;

        // Move file immediately
        $path = $this->saveToPublic($file, $folder, $fileName);

        // Get size from the moved file, not temp file
        $size = filesize(public_path($path));

        $relationMethod = 'cursos_materiales_' . $type;
        $cursoMaterial->$relationMethod()->create([
            $urlField => $path,
            'id_curso_material' => $cursoMaterial->id,
            'original_name' => $originalName,
            'size' => $size,
            'mime_type' => $mimeType,
        ]);
    }
    /**
     * Maneja los archivos marcados para eliminación usando modelos directamente
     */
    protected function handleDeletedFiles(CursoMaterialEditRequest $request, CursoMaterial $cursoMaterial): void
    {
        if ($request->filled('deleted_video_ids')) {
            $cursoMaterial
                ->cursos_materiales_videos()
                ->whereIn('id', $request->deleted_video_ids)
                ->each(function ($video) {
                    if (file_exists(public_path($video->url_video))) {
                        unlink(public_path($video->url_video));
                    }
                    $video->delete();
                });
        }

        // Eliminar imágenes marcadas
        if ($request->filled('deleted_imagen_ids')) {
            $cursoMaterial
                ->cursos_materiales_imagenes()
                ->whereIn('id', $request->deleted_imagen_ids)
                ->each(function ($imagen) {
                    if (file_exists(public_path($imagen->url_imagen))) {
                        unlink(public_path($imagen->url_imagen));
                    }
                    $imagen->delete();
                });
        }

        // Eliminar PDFs marcados
        if ($request->filled('deleted_pdf_ids')) {
            $cursoMaterial
                ->cursos_materiales_pdfs()
                ->whereIn('id', $request->deleted_pdf_ids)
                ->each(function ($pdf) {
                    if (file_exists(public_path($pdf->url_pdf))) {
                        unlink(public_path($pdf->url_pdf));
                    }
                    $pdf->delete();
                });
        }
    }

    /**
     * Maneja los nuevos archivos subidos
     */
    protected function handleNewFiles(CursoMaterialEditRequest $request, CursoMaterial $cursoMaterial): void
    {
        try {
            if ($request->hasFile('pdfs')) {
                foreach ($request->file('pdfs') as $pdf) {
                    $this->storeFile($cursoMaterial, $pdf, 'pdfs', 'url_pdf');
                }
            }

            // Procesar nuevos Videos
            if ($request->hasFile('videos')) {
                foreach ($request->file('videos') as $video) {
                    $this->storeFile($cursoMaterial, $video, 'videos', 'url_video');
                }
            }

            // Procesar nuevas Imágenes
            if ($request->hasFile('imagenes')) {
                foreach ($request->file('imagenes') as $imagen) {
                    $this->storeFile($cursoMaterial, $imagen, 'imagenes', 'url_imagen');
                }
            }
        } catch (\Exception $e) {
            Log::error('Error al procesar nuevos archivos: ' . $e->getMessage());
            throw $e;
        }
    }

    /**
     * Almacena un archivo y lo asocia al material del curso
     */
    protected function storeFile(CursoMaterial $cursoMaterial, $file, string $type, string $urlField): void
    {
        if (!$file->isValid()) {
            throw new \RuntimeException('Archivo inválido o corrupto: ' . $file->getClientOriginalName());
        }

        if (!file_exists($file->getPathname())) {
            throw new \RuntimeException('El archivo temporal ya no está disponible: ' . $file->getClientOriginalName());
        }

        $originalName = $file->getClientOriginalName();
        $fileName = $this->generateFileName($file);
        $folder = 'materiales/' . $type;
        $path = $this->saveToPublic($file, $folder, $fileName);

        $relationMethod = 'cursos_materiales_' . $type;

        $cursoMaterial->$relationMethod()->create([
            $urlField => $path,
            'id_curso_material' => $cursoMaterial->id,
            'original_name' => $originalName,
            'size' => $file->getSize(),
            'mime_type' => $file->getMimeType(),
        ]);
    }

    /**
     * Muestra la vista de materiales a crear.
     *
     * @param int $id Curso ID
     * @return Response
     */
    public function edit(int $id): Response
    {
        $material = CursoMaterial::with('curso', 'cursos_materiales_videos', 'cursos_materiales_imagenes', 'cursos_materiales_pdfs', 'cursos_materiales_redes')->findOrFail($id);
        $cursos = Curso::all();
        return Inertia::render('CursosMateriales/Edit', [
            'curso_material' => $material,
            'cursos' => $cursos,
        ]);
    }

    /**
     * Elimina todo el material asociado a un curso material.
     *
     * @param int $id ID del material del curso
     * @return \Illuminate\Http\RedirectResponse
     */
    public function destroy(int $id)
    {
        try {
            // Encontrar el material del curso con todas sus relaciones
            $cursoMaterial = CursoMaterial::with(['cursos_materiales_pdfs', 'cursos_materiales_videos', 'cursos_materiales_imagenes'])->findOrFail($id);

            // Eliminar archivos físicos y registros de la base de datos
            $this->deleteRelationFiles($cursoMaterial->cursos_materiales_pdfs, 'url_pdf');
            $this->deleteRelationFiles($cursoMaterial->cursos_materiales_videos, 'url_video');
            $this->deleteRelationFiles($cursoMaterial->cursos_materiales_imagenes, 'url_imagen');

            // Eliminar el registro principal
            $cursoMaterial->delete();

            return redirect()->route('cursos-materiales.index')->with('success', 'Material eliminado correctamente.');
        } catch (\Exception $e) {
            Log::error('Error al eliminar material: ' . $e->getMessage(), [
                'exception' => $e,
                'material_id' => $id,
            ]);

            return redirect()
                ->back()
                ->with('error', 'Error al eliminar el material: ' . $e->getMessage());
        }
    }
}
