<?php

namespace App\Http\Controllers;

use App\Http\Requests\CursoEstudianteRequest;
use App\Models\Curso;
use App\Models\CursoEstudiante;
use App\Models\Estudiante;
use Inertia\Inertia;
use Inertia\Response;


/**
 * Controlador CursoEstudianteController
 *
 * Maneja la asignación de estudiantes a cursos usando Inertia.js.
 */
class CursoEstudianteController extends Controller
{
    /**
     * Muestra la lista de asignaciones de estudiantes a cursos.
     *
     * @return Response
     */
    public function index(): Response
    {

        $cursosEstudiantes = CursoEstudiante::with(['estudiante', 'curso'])->latest()->get();

        return Inertia::render('CursosEstudiantes/Index', [
            'cursosEstudiantes' => $cursosEstudiantes,
            'estudiantes' => Estudiante::all(),
            'cursos' => Curso::all(),
        ]);
    }

    /**
     * Asigna un estudiante a un curso.
     *
     * @param CursoEstudianteRequest $request
     * @return \Illuminate\Http\RedirectResponse
     */
    public function store(CursoEstudianteRequest $request)
    {
        CursoEstudiante::create($request->validated());

        return redirect()->back()->with('success', 'Estudiante asignado al curso exitosamente.');
    }

    /**
     * Actualiza la asignación de un estudiante a un curso.
     *
     * @param CursoEstudianteRequest $request
     * @param CursoEstudiante $cursoEstudiante
     * @return \Illuminate\Http\RedirectResponse
     */
    public function update(CursoEstudianteRequest $request, CursoEstudiante $cursoEstudiante)
    {
        $cursoEstudiante->update($request->validated());

        return redirect()->back()->with('success', 'Asignación actualizada exitosamente.');
    }

    /**
     * Elimina una asignación de estudiante a curso.
     *
     * @param CursoEstudiante $cursoEstudiante
     * @return \Illuminate\Http\RedirectResponse
     */
    public function destroy(CursoEstudiante $cursoEstudiante)
    {
        $cursoEstudiante->delete();

        return redirect()->back()->with('success', 'Asignación eliminada exitosamente.');
    }
}
