<?php

namespace App\Http\Controllers;

use App\Http\Requests\CursoRequest;
use App\Models\Curso;
use Inertia\Inertia;
use Inertia\Response;

/**
 * Controlador CursoController
 *
 * Maneja la gestión de cursos usando Inertia.js.
 */
class CursoController extends Controller
{
    /**
     * Muestra la lista de cursos.
     *
     * @return Response
     */
    public function index()
    {
        $perPage = request('per_page', 10);
        $search = request('search', '');

        $cursos = Curso::latest()
            ->where('nombre', 'like', "%{$search}%")
            ->paginate($perPage)
            ->appends(['search' => $search, 'per_page' => $perPage]);

        return Inertia::render('Cursos/Index', [
            'cursos' => $cursos,
            'filters' => request()->only(['search', 'per_page']),
            'status' => session()->only(['success', 'error']),
        ]);
    }

    /**
     * Almacena un nuevo curso en la base de datos.
     *
     * @param CursoRequest $request
     * @return \Illuminate\Http\RedirectResponse
     */
    public function store(CursoRequest $request)
    {
        $curso = Curso::create($request->validated());
        return redirect()->back()->with('success', 'Curso: ' . $curso->nombre . ' creado exitosamente.');
    }

    /**
     * Actualiza un curso existente en la base de datos.
     *
     * @param CursoRequest $request
     * @param Curso $curso
     * @return \Illuminate\Http\RedirectResponse
     */
    public function update(CursoRequest $request, Curso $curso)
    {
        $curso->update($request->validated());

        return redirect()->back()->with('success', 'Curso: ' . $curso->nombre . ' actualizado exitosamente.');
    }

    /**
     * Elimina un curso de la base de datos.
     *
     * @param Curso $curso
     * @return \Illuminate\Http\RedirectResponse
     */
    public function destroy(Curso $curso)
    {
        $curso->delete();

        return redirect()->back()->with('success', 'Curso eliminado exitosamente.');
    }
}
