<?php

namespace App\Http\Controllers;

use App\Http\Requests\ConductaRequest;
use App\Models\Conducta;
use Inertia\Inertia;
use Inertia\Response;

/**
 * Controlador ConductaController
 *
 * Maneja la gestión de conductas usando Inertia.js.
 */
class ConductaController extends Controller
{
    /**
     * Muestra la lista de conductas.
     *
     * @return Response
     */
    public function index()
    {
        $perPage = request('per_page', 10);
        $search = request('search', '');

        $conductas = Conducta::latest()
            ->where('fecha', 'like', "%{$search}%")
            ->orWhere('tipo', 'like', "%{$search}%")
            ->paginate($perPage)
            ->appends(['search' => $search, 'per_page' => $perPage]);

        return Inertia::render('Conductas/Index', [
            'conductas' => $conductas,
            'filters' => request()->only(['search', 'per_page']),
            'status' => session()->only(['success', 'error']),
        ]);
    }

    /**
     * Almacena un nuevo conducta en la base de datos.
     *
     * @param ConductaRequest $request
     * @return \Illuminate\Http\RedirectResponse
     */
    public function store(ConductaRequest $request)
    {
        $conducta = Conducta::create($request->validated());
        return redirect()->back()->with('success', 'Conducta: ' . $conducta->nombre . ' creado exitosamente.');
    }

    /**
     * Actualiza un conducta existente en la base de datos.
     *
     * @param ConductaRequest $request
     * @param int $id
     * @return \Illuminate\Http\RedirectResponse
     */
    public function update(ConductaRequest $request, int $id)
    {
        $conducta = Conducta::findOrFail($id);
    
        $conducta->update($request->validated());

        return redirect()->back()->with('success', 'Conducta: ' . $conducta->nombre . ' actualizado exitosamente.');
    }

    /**
     * Elimina un conducta de la base de datos.
     *
     * @param int $id conducta
     * @return \Illuminate\Http\RedirectResponse
     */
    public function destroy(int $id)
    {
        $conducta = Conducta::findOrFail($id);
        if (!$conducta) {
            return redirect()->back()->with('error', 'Conducta no encontrada.');
        }
        $conducta->delete();

        return redirect()->back()->with('success', 'Conducta eliminado exitosamente.');
    }
}
