<?php

namespace App\Http\Controllers;

use App\Models\Nota;
use App\Models\Conducta;
use App\Models\Asistencia;
use App\Models\CursoMaterial;
use Inertia\Inertia;
use Inertia\Response;
use Illuminate\Http\Request;
use Illuminate\Support\Facades\Auth;

/**
 * Controlador CalificacionController
 *
 * Maneja la gestión de calificaciones usando Inertia.js.
 */
class CalificacionEstudianteController extends Controller
{
    /**
     * Muestra la lista de calificaciones.
     *
     * @param Request $request
     * @return Response
     */
    public function index(Request $request)
    {
        $estudiante = Auth::guard('estudiante')->user();

        if (!$estudiante) {
            return redirect()->back()->with('error', 'Estudiante no encontrado.');
        }

        $notaPerPage = $request->query('nota')['per_page'] ?? 10;
        $notaSearch = $request->query('nota')['search'] ?? null;
        $notaPage = $request->query('nota')['page'] ?? 1;

        $notas = Nota::where('id_estudiante', $estudiante->id)
            ->when(!empty($notaSearch), function ($query) use ($notaSearch) {
                $query->where(function ($subquery) use ($notaSearch) {
                    $subquery->where('fecha', 'like', "%{$notaSearch}%")->orWhere('nota', 'like', "%{$notaSearch}%");
                });
            })
            ->latest()
            ->paginate($notaPerPage, ['*'], 'page', $notaPage)
            ->appends($request->query());

        $conductaPerPage = $request->query('conducta')['per_page'] ?? 10;
        $conductaSearch = $request->query('conducta')['search'] ?? null;
        $conductaPage = $request->query('conducta')['page'] ?? 1;

        $conductas = Conducta::where('id_estudiante', $estudiante->id)
            ->when(!empty($conductaSearch), function ($query) use ($conductaSearch) {
                $query->where(function ($subquery) use ($conductaSearch) {
                    $subquery
                        ->where('fecha', 'like', "%{$conductaSearch}%")
                        ->orWhere('conducta', 'like', "%{$conductaSearch}%")
                        ->orWhere('observacion', 'like', "%{$conductaSearch}%");
                });
            })
            ->latest()
            ->paginate($conductaPerPage, ['*'], 'page', $conductaPage)
            ->appends($request->query());

        $asistenciaPerPage = $request->query('asistencia')['per_page'] ?? 10;
        $asistenciaSearch = $request->query('asistencia')['search'] ?? null;
        $asistenciaPage = $request->query('asistencia')['page'] ?? 1;

        $asistencias = Asistencia::where('id_estudiante', $estudiante->id)
            ->when(!empty($asistenciaSearch), function ($query) use ($asistenciaSearch) {
                $query->where(function ($subquery) use ($asistenciaSearch) {
                    $subquery->where('fecha', 'like', "%{$asistenciaSearch}%")->orWhere('tipo', 'like', "%{$asistenciaSearch}%");
                });
            })
            ->latest()
            ->paginate($asistenciaPerPage, ['*'], 'page', $asistenciaPage)
            ->appends($request->query());

        $cursos_materiales = CursoMaterial::with('curso', 'cursos_materiales_videos', 'cursos_materiales_imagenes', 'cursos_materiales_pdfs')->get();

        return Inertia::render('Calificaciones/Index', [
            'notas' => $notas,
            'conductas' => $conductas,
            'asistencias' => $asistencias,
            'cursos_materiales' => $cursos_materiales,
            'estudiante' => $estudiante,
            'filters' => $request->query(),
            'status' => session()->only(['success', 'error']),
        ]);
    }

    public function show(int $id)
    {
        
        $estudiante = Auth::guard('estudiante')->user();

        if (!$estudiante) {
            return redirect()->back()->with('error', 'Estudiante no encontrado.');
        }
        $curso_material = CursoMaterial::with('curso', 'cursos_materiales_videos', 'cursos_materiales_imagenes', 'cursos_materiales_pdfs', 'cursos_materiales_redes')->findOrFail($id);
        return Inertia::render('Calificaciones/Show', [
            'curso_material' => $curso_material,
            'estudiante' => $estudiante,
        ]);
    }
}
