<?php

namespace App\Http\Controllers;

use App\Http\Requests\AsistenciaRequest;
use App\Models\Asistencia;
use Inertia\Inertia;
use Inertia\Response;

/**
 * Controlador AsistenciaController
 *
 * Maneja la gestión de asistencias usando Inertia.js.
 */
class AsistenciaController extends Controller
{
    /**
     * Muestra la lista de asistencias.
     *
     * @return Response
     */
    public function index()
    {
        $perPage = request('per_page', 10);
        $search = request('search', '');

        $asistencias = Asistencia::latest()
            ->where('fecha', 'like', "%{$search}%")
            ->orWhere('tipo', 'like', "%{$search}%")
            ->paginate($perPage)
            ->appends(['search' => $search, 'per_page' => $perPage]);

        return Inertia::render('Asistencias/Index', [
            'asistencias' => $asistencias,
            'filters' => request()->only(['search', 'per_page']),
            'status' => session()->only(['success', 'error']),
        ]);
    }

    /**
     * Almacena un nuevo asistencia en la base de datos.
     *
     * @param AsistenciaRequest $request
     * @return \Illuminate\Http\RedirectResponse
     */
    public function store(AsistenciaRequest $request)
    {
        $asistencia = Asistencia::create($request->validated());
        return redirect()->back()->with('success', 'Asistencia: ' . $asistencia->nombre . ' creado exitosamente.');
    }

    /**
     * Actualiza un asistencia existente en la base de datos.
     *
     * @param AsistenciaRequest $request
     * @param int $id
     * @return \Illuminate\Http\RedirectResponse
     */
    public function update(AsistenciaRequest $request, int $id)
    {
        $asistencia = Asistencia::findOrFail($id);
    
        $asistencia->update($request->validated());

        return redirect()->back()->with('success', 'Asistencia: ' . $asistencia->nombre . ' actualizado exitosamente.');
    }

    /**
     * Elimina un asistencia de la base de datos.
     *
     * @param int $id asistencia
     * @return \Illuminate\Http\RedirectResponse
     */
    public function destroy(int $id)
    {
        $asistencia = Asistencia::findOrFail($id);
        if (!$asistencia) {
            return redirect()->back()->with('error', 'Asistencia no encontrada.');
        }
        $asistencia->delete();

        return redirect()->back()->with('success', 'Asistencia eliminado exitosamente.');
    }
}
