<?php

namespace App\Http\Controllers;

use App\Http\Requests\ResultadoEncuestaRequest;
use App\Models\PivoteEncuestaEstudiante;
use Illuminate\Http\JsonResponse;
use Illuminate\Support\Collection;

class ResultadoEncuestaController extends Controller
{
    public function getAll(ResultadoEncuestaRequest $request): JsonResponse
    {
        $idCiclo = $request->input('id_ciclo');

        $resultados_encuesta = PivoteEncuestaEstudiante::with([
            'pivote_encuesta_docente.pivote_pregunta_docente.pivote_area_curso_docente.docente.persona',
            'pivote_encuesta_docente.pivote_pregunta_docente.pregunta',
            'pivote_horario_carrera_estudiante.estudiante.persona',
            'pivote_horario_carrera_estudiante.pivote_horario.ciclo'
        ])
            ->whereHas('pivote_horario_carrera_estudiante.pivote_horario.ciclo', function ($query) use ($idCiclo) {
                if ($idCiclo) {
                    $query->where('id', $idCiclo);
                } else {
                    $query->where('estado', 'VIGENTE');
                }
            })
            ->get()
            ->groupBy(fn($resultado) => optional($resultado->pivote_encuesta_docente->pivote_pregunta_docente->pivote_area_curso_docente->docente)->id)
            ->map(function (Collection $docenteResultados) use ($idCiclo) {
                $docente = $docenteResultados->first()->pivote_encuesta_docente->pivote_pregunta_docente->pivote_area_curso_docente->docente ?? null;

                if ($docente && !$docente->relationLoaded('persona')) {
                    $docente->load('persona');
                }

                // Lista de estudiantes que ya han realizado la encuesta
                $estudiantes_encuestados = $docenteResultados->map(fn($resultado) => $resultado->pivote_horario_carrera_estudiante->estudiante)->unique('id');

                // Obtener todos los estudiantes inscritos en el ciclo actual
                $estudiantes_inscritos = \App\Models\PivoteHorarioCarreraEstudiante::with('estudiante.persona')
                    ->whereHas('pivote_horario.ciclo', function ($query) use ($idCiclo) {
                        if ($idCiclo) {
                            $query->where('id', $idCiclo);
                        } else {
                            $query->where('estado', 'VIGENTE');
                        }
                    })
                    ->get()
                    ->map(fn($pivote) => $pivote->estudiante)
                    ->unique('id');

                // Filtrar los estudiantes que aún no han realizado la encuesta
                $estudiantes_por_encuestar = $estudiantes_inscritos->diff($estudiantes_encuestados);

                $totalRespuestas = $docenteResultados->sum('respuesta');
                $cantidadRespuestas = $docenteResultados->count();
                $promedio = $cantidadRespuestas > 0 ? $totalRespuestas / $cantidadRespuestas : 0;

                return [
                    'docente' => $docente,
                    'estudiantes_encuestados' => $estudiantes_encuestados,
                    'estudiantes_por_encuestar' => $estudiantes_por_encuestar, // Ahora retorna solo los que faltan
                    'promedio' => $promedio,
                ];
            })
            ->values();
        return response()->json([
            'resultados_encuesta' => $resultados_encuesta,
        ], 200);
    }
}
