<?php

namespace App\Http\Controllers;

use App\Http\Requests\PivoteHorarioCarreraEstudianteRequest;
use App\Models\Credencial;
use App\Models\Evaluacion\Conducta;
use App\Models\Persona\Apoderado;
use App\Models\Persona\Estudiante;
use App\Models\Persona\Persona;
use App\Models\PivoteHorarioCarreraEstudiante;
use App\Models\User;
use App\Services\SearchHorarioCarreraEstudiante;
use Carbon\Carbon;
use Illuminate\Http\JsonResponse;
use Illuminate\Http\Request;

class PivoteHorarioCarreraEstudianteController extends Controller
{
    protected $searchHorarioCarreraEstudiante;

    public function __construct(SearchHorarioCarreraEstudiante $searchHorarioCarreraEstudiante)
    {
        $this->searchHorarioCarreraEstudiante = $searchHorarioCarreraEstudiante;
    }

    public function getAll(Request $request): JsonResponse
    {
        $query = PivoteHorarioCarreraEstudiante::query();
        $pagination = $request->input('pagination', 10);
        $search = $request->input('search');

        if ($search) {
            $query = $this->searchHorarioCarreraEstudiante->buscarCarreraHorarioDocente($search, null);
        }

        $pivote_horarios_carreras_estudiantes = $query->with('estudiante.persona', 'estudiante.colegio', 'estudiante.apoderado.persona', 'carrera', 'pivote_horario.ciclo', 'pivote_horario.horario', 'pagos', 'becas', 'asistencias', 'notas')->latest()->paginate($pagination)->appends($request->query());

        return response()->json(
            [
                'pivote_horarios_carreras_estudiantes' => $pivote_horarios_carreras_estudiantes,
                'params' => $request->query() ?: null,
            ],
            200,
        );
    }

    public function show($id): JsonResponse
    {
        $pivote_horario_carrera_estudiante = PivoteHorarioCarreraEstudiante::findOrFail($id);

        if (!$pivote_horario_carrera_estudiante) {
            return response()->json(['message' => 'PivoteHorarioCarreraEstudiante no encontrado.'], 404);
        }

        return response()->json(
            [
                'pivote_horario_carrera_estudiante' => $pivote_horario_carrera_estudiante,
                'message' => 'PivoteHorarioCarreraEstudiante: ' . $pivote_horario_carrera_estudiante->name . ', encontrado exitosamente!.',
            ],
            200,
        );
    }

    public function save(PivoteHorarioCarreraEstudianteRequest $request): JsonResponse
    {
        $personaApoderado = Persona::where('dni', $request->input('estudiante.dni_apoderado'))->first();

        if (!$personaApoderado) {
            return response()->json(['message' => 'El apoderado no existe, no encontrado.'], 404);
        }

        $apoderado = Apoderado::firstOrCreate(['id_persona' => $personaApoderado->id], []);

        $estudiante = Estudiante::updateOrCreate(
            ['id_persona' => $request->input('estudiante.id_persona')],
            [
                'observacion' => $request->input('estudiante.observacion'),
                'estado_estudiante' => 'ESTUDIANTE',
                'fotografia' => $request->input('estudiante.fotografia'),
                'id_colegio' => $request->input('estudiante.id_colegio'),
                'id_apoderado' => $apoderado->id,
            ],
        );

        $pivoteHorarioCarreraEstudiante = PivoteHorarioCarreraEstudiante::create([
            'id_pivote_horario' => $request->input('id_pivote_horario'),
            'id_carrera' => $request->input('id_carrera'),
            'id_estudiante' => $estudiante->id,
            'estado_pago' => $request->input('estado_pago') ?? 'FALTA PAGO',
            'id_universidad' => $request->input('id_universidad'),
            'estado_mensaje' => $request->input('estado_mensaje') ?? 'NO ENVIADO',
        ]);

        $pivoteHorarioCarreraEstudiante->load('estudiante');

        Conducta::create([
            'conducta' => 'MUY BUENA',
            'observacion' => '',
            'fecha_conducta' => Carbon::now(),
            'id_pivote_horario_carrera_estudiante' => $pivoteHorarioCarreraEstudiante->id,
        ]);

        $OtherDni = Credencial::where('dni', $estudiante->persona->dni)->first();

        if (!$OtherDni) {
            Credencial::create([
                'usuario' => $estudiante->persona->dni ?? 'User' . $estudiante->id,
                'password' => bcrypt($estudiante->persona->dni ?? 'User' . $estudiante->id),
                'dni' => $estudiante->persona->dni ?? 'User' . $estudiante->id,
            ]);
        }

        return response()->json(
            [
                'pivote_horario_carrera_estudiante' => $pivoteHorarioCarreraEstudiante,
                'message' => '¡El registro se creó correctamente!',
            ],
            201,
        );
    }

    public function update($id, PivoteHorarioCarreraEstudianteRequest $request): JsonResponse
    {
        $pivoteHorarioCarreraEstudiante = PivoteHorarioCarreraEstudiante::findOrFail($id);

        $personaApoderado = Persona::where('dni', $request->input('estudiante.dni_apoderado'))->first();

        if (!$personaApoderado) {
            return response()->json(['message' => 'El apoderado no existe.'], 404);
        }

        $apoderado = Apoderado::firstOrCreate(['id_persona' => $personaApoderado->id]);

        $estudiante = Estudiante::updateOrCreate(
            ['id_persona' => $request->input('estudiante.id_persona')],
            [
                'observacion' => $request->input('estudiante.observacion'),
                'estado_estudiante' => $request->input('estudiante.estado_estudiante'),
                'fotografia' => $request->input('estudiante.fotografia'),
                'id_colegio' => $request->input('estudiante.id_colegio'),
                'id_apoderado' => $apoderado->id,
            ],
        );

        $pivoteHorarioCarreraEstudiante->update([
            'id_pivote_horario' => $request->input('id_pivote_horario'),
            'id_carrera' => $request->input('id_carrera'),
            'id_estudiante' => $estudiante->id,
            'id_universidad' => $request->input('id_universidad'),
            'estado_pago' => $request->input('estado_pago'),
            'estado_mensaje' => $request->input('estado_mensaje'),
        ]);

        $pivoteHorarioCarreraEstudiante->load('estudiante');

        $OtherDni = Credencial::where('dni', $estudiante->persona->dni)->first();

        if (!$OtherDni) {
            Credencial::create([
                'usuario' => $estudiante->persona->dni ?? 'User' . $estudiante->id,
                'password' => bcrypt($estudiante->persona->dni ?? 'User' . $estudiante->id),
                'dni' => $estudiante->persona->dni ?? 'User' . $estudiante->id,
            ]);
        }

        return response()->json(
            [
                'pivote_horario_carrera_estudiante' => $pivoteHorarioCarreraEstudiante,
                'message' => 'El registro se ha actualizado exitosamente.',
            ],
            200,
        );
    }

    public function delete($id): JsonResponse
    {
        $pivote_horario_carrera_estudiante = PivoteHorarioCarreraEstudiante::findOrFail($id);

        if (!$pivote_horario_carrera_estudiante) {
            return response()->json(['message' => 'Estudiante no encontrado.'], 404);
        }

        if ($pivote_horario_carrera_estudiante->conductas()->exists()) {
            $pivote_horario_carrera_estudiante->conductas()->delete();
        }

        $pivote_horario_carrera_estudiante->delete();

        return response()->json(
            [
                'message' => 'El pivote_horario_carrera_estudiante: ' . $pivote_horario_carrera_estudiante->id . ', se ha eliminado exitosamente!',
            ],
            200
        );
    }
}
