<?php

namespace App\Http\Controllers;

use App\Http\Requests\PivoteAreaCursoDocenteRequest;
use App\Models\Academico\Curso;
use App\Models\PivoteAreaCursoDocente;
use App\Services\SearchAreaCursoDocente;
use Illuminate\Http\JsonResponse;
use Illuminate\Http\Request;
use Illuminate\Database\Eloquent\Builder;

class PivoteAreaCursoDocenteController extends Controller
{
    protected $searchAreaCursoDocente;

    public function __construct(SearchAreaCursoDocente $searchAreaCursoDocente)
    {
        $this->searchAreaCursoDocente = $searchAreaCursoDocente;
    }

    public function paramDocente(Builder $query, string $docente): Builder
    {
        if ($docente === 'VIGENTE') {
            $query->whereHas('docente', function ($q) use ($docente) {
                $q->where('estado', 'like', "%$docente%");
            });
        }
        return $query;
    }

    public function getAll(Request $request): JsonResponse
    {
        $query = PivoteAreaCursoDocente::query();
        $pagination = $request->input('pagination', 10);
        $search = $request->input('search');
        $params = $request->input('param', []);

        if ($search) {
            $query = $this->searchAreaCursoDocente->buscarAreaCursoDocente($search);
        }
        if (!empty($params['docente'])) {
            $query = $this->paramDocente($query, $params['docente']);
        }
        $pivote_areas_cursos_docentes = $query->with('pivote_preguntas_docentes', 'area', 'curso', 'docente.persona')->latest()->paginate($pagination)->appends($request->query());

        return response()->json(
            [
                'pivote_areas_cursos_docentes' => $pivote_areas_cursos_docentes,
                'params' => $request->query() ?: null,
            ],
            200,
        );
    }

    public function show($id): JsonResponse
    {
        $pivote_curso_area_docente = PivoteAreaCursoDocente::findOrFail($id);

        if (!$pivote_curso_area_docente) {
            return response()->json(['message' => 'El curso no encontrado.'], 404);
        }
        $pivote_curso_area_docente->load('area', 'curso', 'docente.persona');
        return response()->json(
            [
                'pivote_curso_area_docente' => $pivote_curso_area_docente,
                'message' => 'El curso: ' . $pivote_curso_area_docente->curso->nombre . ', encontrado exitosamente!.',
            ],
            200,
        );
    }

    public function save(PivoteAreaCursoDocenteRequest $request): JsonResponse
    {
        $areas = $request->areas;
        $pivote_curso_area_docente = [];

        foreach ($areas as $area) {
            $pivote_curso_area_docente = PivoteAreaCursoDocente::create([
                'id_curso' => $request->id_curso,
                'id_docente' => $request->id_docente,
                'id_area' => $area['id'],
            ]);
        }
        return response()->json(
            [
                'pivote_curso_area_docente' => $pivote_curso_area_docente,
                'message' => 'El curso: ' . $pivote_curso_area_docente->curso->nombre . ', se ha creado exitosamente!.',
            ],
            201,
        );
    }

    public function update($id, PivoteAreaCursoDocenteRequest $request): JsonResponse
    {

        $curso = Curso::findOrFail($request->id_curso);

        if (!$curso) {
            return response()->json(['message' => 'Curso no encontrado.'], 404);
        }

        $areas = $request->areas ?? [];
        $areaIds = collect($areas)->pluck('id')->toArray();

        $currentAreaIds = $curso->pivote_areas_cursos_docentes->pluck('id_area')->toArray();
        $currentAreas = $curso->pivote_areas_cursos_docentes->keyBy('id_area');
        $areasToDelete = array_diff($currentAreaIds, $areaIds);
        $areasToAdd = array_diff($areaIds, $currentAreaIds);

        if (!empty($areasToDelete)) {
            PivoteAreaCursoDocente::where('id_curso', $curso->id)
                ->whereIn('id_area', $areasToDelete)
                ->delete();
        }

        foreach ($areasToAdd as $areaId) {
            PivoteAreaCursoDocente::updateOrCreate([
                'id_curso' => $curso->id,
                'id_area' => $areaId,
                'id_docente' => $request->id_docente,
            ]);
        }

        foreach ($currentAreas as $areaId => $pivote) {
            if ($pivote->id_docente !== $request->id_docente) {
                $pivote->update(['id_docente' => $request->id_docente]);
            }
        }

        $curso->load('pivote_areas_cursos_docentes.area', 'pivote_areas_cursos_docentes.docente.persona');

        return response()->json(
            [
                'curso' => $curso,
                'message' => 'El curso: ' . $curso->nombre . ' y sus áreas asociadas se han actualizado exitosamente.',
            ],
            200,
        );
    }

    public function delete($id): JsonResponse
    {
        $pivote_curso_area_docente = PivoteAreaCursoDocente::findOrFail($id);

        if (!$pivote_curso_area_docente) {
            return response()->json(['message' => 'curso no encontrado.'], 404);
        }

        $pivote_curso_area_docente->delete();

        return response()->json(
            [
                'message' => 'El curso: ' . $pivote_curso_area_docente->curso->nombre . ', se ha eliminado exitosamente!.',
            ],
            200,
        );
    }
}
