<?php

namespace App\Http\Controllers\Persona;

use App\Http\Controllers\Controller;
use App\Http\Requests\Persona\EstudianteRequest;
use App\Models\Persona\Estudiante;
use Illuminate\Database\Eloquent\Builder;
use Illuminate\Http\JsonResponse;
use Illuminate\Http\Request;

class EstudianteController extends Controller
{
 
    private function searchEstudiante(Builder $query, string $search): Builder
    {
        return $query->where('observacion', 'like', "%$search%")->orWhere('estado_estudiante', 'like', "%$search%");
    }

    private function filterByEstado(Builder $query,string $estado): Builder
    {
        return $query->where('estado_estudiante', $estado);
    }

    public function getAll(Request $request): JsonResponse
    {
        $query = Estudiante::query();

        $pagination = $request->input('pagination', 10);
        $search = $request->input('search');
        $estado = $request->input('param.estado');
        
        if ($search) {
            $query = $this->searchEstudiante($query, $search);
        }

        if ($estado) {
            $query = $this->filterByEstado($query, $estado);
        }
        

        $estudiantes = $query->with('persona', 'colegio', 'apoderado.persona', 'pivote_horarios_carreras_estudiantes.carrera', 'pivote_horarios_carreras_estudiantes.pivote_horario.ciclo', 'pivote_horarios_carreras_estudiantes.pagos')->paginate($pagination)->appends($request->query());

        return response()->json(
            [
                'estudiantes' => $estudiantes,
                'params' => $request->query() ?: null,
            ],
            200,
        );
    }

    public function show($id): JsonResponse
    {
        $estudiante = Estudiante::findOrFail($id);

        if (!$estudiante) {
            return response()->json(['message' => 'Estudiante no encontrado.'], 404);
        }
        $estudiante->load('Persona');
        return response()->json(
            [
                'estudiante' => $estudiante,
                'message' => 'Estudiante: ' . $estudiante->nombre . ', encontrado exitosamente!.',
            ],
            200,
        );
    }

    public function save(EstudianteRequest $request): JsonResponse
    {
        $estudiante = Estudiante::create($request->validated());
        $estudiante->save();
        $estudiante->load('Persona');
        return response()->json(
            [
                'estudiante' => $estudiante,
                'message' => 'El estudiante: ' . $estudiante->nombre . ', se ha creado exitosamente!.',
            ],
            201,
        );
    }

    public function update($id, EstudianteRequest $request): JsonResponse
    {
        $estudiante = Estudiante::findOrFail($id);

        if (!$estudiante) {
            return response()->json(['message' => 'El estudiante' . $request->nombre . ', no encontrado.'], 404);
        }

        $estudiante->update($request->validated());
        $estudiante->save();
        $estudiante->load('Persona');

        return response()->json(
            [
                'estudiante' => $estudiante,
                'message' => 'El estudiante: ' . $estudiante->nombre . ', se ha actualizado exitosamente!.',
            ],
            200,
        );
    }

    public function delete($id): JsonResponse
    {
        $estudiante = Estudiante::findOrFail($id);

        if (!$estudiante) {
            return response()->json(['message' => 'Estudiante no encontrado.'], 404);
        }

        $estudiante->delete();

        return response()->json(
            [
                'message' => 'El estudiante: ' . $estudiante->nombre . ', se ha eliminado exitosamente!.',
            ],
            200,
        );
    }
}
