<?php

namespace App\Http\Controllers\Persona;

use App\Http\Controllers\Controller;
use App\Http\Requests\Persona\DocenteRequest;
use App\Models\Persona\Docente;
use Illuminate\Database\Eloquent\Builder;
use Illuminate\Http\JsonResponse;
use Illuminate\Http\Request;
use App\Services\SearchDocente;

class DocenteController extends Controller
{
    protected $searchDocente;

    public function __construct(SearchDocente $searchDocente)
    {
        $this->searchDocente = $searchDocente;
    }

    public function getAll(Request $request): JsonResponse
    {
        $query = Docente::query();

        $pagination = $request->input('pagination', 10);
        $search = $request->input('search');

        if ($search) {
            $query = $this->searchDocente->buscarDocentes($search);
        }

        $docentes = $query->with('pivote_areas_cursos_docentes.pivote_preguntas_docentes.pregunta', 'pivote_areas_cursos_docentes.curso', 'persona')->latest()->paginate($pagination)->appends($request->query());

        return response()->json(
            [
                'docentes' => $docentes,
                'params' => $request->query() ?: null,
            ],
            200,
        );
    }

    public function show($id): JsonResponse
    {
        $docente = Docente::findOrFail($id);

        if (!$docente) {
            return response()->json(['message' => 'Docente no encontrado.'], 404);
        }
        $docente->load('persona');
        return response()->json(
            [
                'docente' => $docente,
                'message' => 'Docente: ' . $docente->nombre . ', encontrado exitosamente!.',
            ],
            200,
        );
    }

    public function save(DocenteRequest $request): JsonResponse
    {
        $docente = Docente::create($request->validated());
        $docente->save();
        $docente->load('persona');
        return response()->json(
            [
                'docente' => $docente,
                'message' => 'El docente: ' . $docente->nombre . ', se ha creado exitosamente!.',
            ],
            201,
        );
    }

    public function update($id, DocenteRequest $request): JsonResponse
    {
        $docente = Docente::findOrFail($id);

        if (!$docente) {
            return response()->json(['message' => 'El docente' . $request->nombre . ', no encontrado.'], 404);
        }

        $docente->update($request->validated());
        $docente->save();
        $docente->load('persona');

        return response()->json(
            [
                'docente' => $docente,
                'message' => 'El docente: ' . $docente->nombre . ', se ha actualizado exitosamente!.',
            ],
            200,
        );
    }

    public function delete($id): JsonResponse
    {
        $docente = Docente::findOrFail($id);

        if (!$docente) {
            return response()->json(['message' => 'Docente no encontrado.'], 404);
        }

        $docente->delete();

        return response()->json(
            [
                'message' => 'El docente: ' . $docente->nombre . ', se ha eliminado exitosamente!.',
            ],
            200,
        );
    }
}
