<?php

namespace App\Http\Controllers;

use App\Http\Requests\InformacionEstudianteRequest;
use App\Models\Evaluacion\Encuesta;
use App\Models\Persona\Persona;

class InformacionEstudianteController extends Controller
{
    /**
     * Buscar personas por DNI.
     *
     * @param string $dni
     * @return \Illuminate\Support\Collection
     */
    private function buscarPersonaPorDni($dni)
    {
        return Persona::with(['estudiante.pivote_horarios_carreras_estudiantes.pivote_horario.ciclo'])
            ->where('dni', $dni)
            ->latest()
            ->first();
    }

    /**
     * Buscar encuesta vigente.
     *
     * @return \Illuminate\Support\Collection
     */
    private function buscarEncuesta($pivote_horario_carrera_estudiante)
    {
        $encuesta = Encuesta::with(['pivote_encuestas_docentes.pivote_pregunta_docente.pivote_area_curso_docente.docente.persona', 'pivote_encuestas_docentes.pivote_pregunta_docente.pivote_area_curso_docente.curso', 'pivote_encuestas_docentes.pivote_pregunta_docente.pregunta', 'pivote_encuestas_docentes.pivote_encuestas_estudiantes'])
            ->where('estado', 'VIGENTE')
            ->latest()
            ->first();

        if (!$encuesta) {
            return null;
        }

        $encuestaDocentes = $encuesta->pivote_encuestas_docentes->filter(function ($pivote_encuesta_docente) use ($pivote_horario_carrera_estudiante) {
            $area_estudiante_id = $pivote_horario_carrera_estudiante->carrera->area->id;
            $area_docente_id = $pivote_encuesta_docente->pivote_pregunta_docente->pivote_area_curso_docente->area->id;
            return $area_estudiante_id === $area_docente_id;
        });

        $respondioEncuesta = $pivote_horario_carrera_estudiante->pivote_encuesta_estudiante()->exists();

        $disponibilidadEncuesta = $encuesta->pivote_encuestas_docentes->isNotEmpty() ? ($respondioEncuesta ? 'RESPONDIO' : 'FALTA_RESPONDER') : 'NO_HAY_PREGUNTAS';


        return [
            'id' => $encuesta->id,
            'estado' => $encuesta->estado,
            'fecha_final' => $encuesta->fecha_final,
            'fecha_inicio' => $encuesta->fecha_inicio,
            'nombre' => $encuesta->nombre,
            'disponibilidad' => $disponibilidadEncuesta,
            'pivote_encuestas_docentes' => $encuestaDocentes->values(),
        ];
    }

    /**
     * Obtener información de estudiantes.
     *
     * @param InformacionEstudianteRequest $request
     * @return \Illuminate\Http\JsonResponse
     */
    public function getInfoEstudiante(InformacionEstudianteRequest $request)
    {
        // Buscar persona por DNI
        $persona = $this->buscarPersonaPorDni($request->dni);

        if (!$persona) {
            return response()->json(['message' => 'El DNI no fue encontrado.'], 404);
        }

        $persona->load('estudiante.pivote_horarios_carreras_estudiantes.carrera.area.pivote_areas_cursos_docentes.curso', 'estudiante.pivote_horarios_carreras_estudiantes.carrera.area.pivote_areas_cursos_docentes.docente.persona', 'estudiante.pivote_horarios_carreras_estudiantes.pivote_horario.horario', 'estudiante.pivote_horarios_carreras_estudiantes.pivote_horario.ciclo', 'estudiante.pivote_horarios_carreras_estudiantes.asistencias', 'estudiante.pivote_horarios_carreras_estudiantes.conductas', 'estudiante.pivote_horarios_carreras_estudiantes.notas', 'estudiante.pivote_horarios_carreras_estudiantes.pagos', 'estudiante.pivote_horarios_carreras_estudiantes.becas', 'apoderado');

        $pivote_horario_carrera_estudiante = $persona->estudiante->pivote_horarios_carreras_estudiantes->last();

        $encuesta = $this->buscarEncuesta($pivote_horario_carrera_estudiante);

        return response()->json(
            [
                'id' => $pivote_horario_carrera_estudiante->id,
                'persona' => $persona,
                'encuesta' => $encuesta,
            ],
            200,
        );
    }
}
