<?php

namespace App\Http\Controllers\Finanza;

use App\Http\Controllers\Controller;
use App\Http\Requests\Finanza\SyncPagoRequest;
use App\Models\Finanza\Pago;
use App\Models\Persona\Estudiante;
use App\Models\PivoteHorarioCarreraEstudiante;
use Illuminate\Http\Request;
use Illuminate\Http\JsonResponse;
use Illuminate\Pagination\LengthAwarePaginator;

class SyncPagoController extends Controller
{

    public function getAll(Request $request): JsonResponse
    {
        $params = $request->input('param', []);
        $pagination = $request->input('pagination', 10);

        $idPivoteEstudiante = $params['id_pivote_horario_carrera_estudiante'] ?? null;
        $pivoteHorarioCarreraEstudiante = PivoteHorarioCarreraEstudiante::find($idPivoteEstudiante);

        if (!$pivoteHorarioCarreraEstudiante) {
            return response()->json(['message' => 'PivoteHorarioCarreraEstudiante no encontrado.'], 404);
        }

        $estudiante = Estudiante::findOrFail($pivoteHorarioCarreraEstudiante->id_estudiante);
        $estudiante->load("pivote_horarios_carreras_estudiantes.pagos");

        // Convertir en colecciones
        $pagosSync = collect($estudiante->pivote_horarios_carreras_estudiantes->pluck('pagos')->flatten());

        // Obtener todos los recibos en un array
        $recibos = [];
        foreach ($pagosSync as $pago) {
            $recibo = $pago->recibo;
            if (!isset($recibos[$recibo])) {
                $recibos[$recibo] = 1;
            } else {
                $recibos[$recibo]++;
            }
        }

        // Filtrar los pagos cuyos recibos NO se repiten
        $pagosNoSincronizados = collect();
        foreach ($pagosSync as $pago) {
            if ($recibos[$pago->recibo] === 1) {
                $pagosNoSincronizados->push($pago);
            }
        }

        $pivoteHorarioCarreraEstudiante->estado_pago = $request->input('estado_pago') ?? 'COMPLETO';
        
        $pivoteHorarioCarreraEstudiante->save();

        // Paginar manualmente la colección
        $currentPage = LengthAwarePaginator::resolveCurrentPage();
        $paginatedPagos = new LengthAwarePaginator(
            $pagosNoSincronizados->forPage($currentPage, $pagination),
            $pagosNoSincronizados->count(),
            $pagination,
            $currentPage,
            ['path' => $request->url(), 'query' => $request->query()]
        );

        return response()->json([
            'sync_pagos' => $paginatedPagos,
            'params' => $request->query() ?: null,
        ], 200);
    }



    public function sync(SyncPagoRequest $request): JsonResponse
    {
        $pagos = $request->input('pagos', []);
        $idPivoteEstudiante = $request->input('id_pivote_horario_carrera_estudiante');

        $pivote = PivoteHorarioCarreraEstudiante::find($idPivoteEstudiante);
        if (!$pivote) {
            return response()->json(['message' => 'El estudiante relacionado no fue encontrado.'], 404);
        }

        $pagosSincronizados = collect();
        foreach ($pagos as $pagoData) {
            $pagoReference = Pago::find($pagoData['id']);
            info("pagoReference".  $pagoReference);
            Pago::create(
                [
                    'tipo' => 'SYNC',
                    'recibo' => $pagoReference['recibo'],
                    'total' => $pagoReference['total'],
                    'id_pivote_horario_carrera_estudiante' => $pivote->id,
                ]
            );
        }

        return response()->json([
            'sync_pagos' => $pagosSincronizados,
            'message' => $pagosSincronizados->count() . ' pagos sincronizados exitosamente!',
        ], 200);
    }
}
