<?php

namespace App\Http\Controllers\Finanza;

use App\Http\Controllers\Controller;
use App\Http\Requests\Finanza\PagoRequest;
use App\Models\Finanza\Pago;
use App\Models\PivoteHorarioCarreraEstudiante;
use Illuminate\Database\Eloquent\Builder;
use Illuminate\Http\JsonResponse;
use Illuminate\Http\Request;

class PagoController extends Controller
{
    private function searchPagos(string $search): Builder
    {
        return Pago::where(function ($query) use ($search) {
            $query->where('recibo', 'like', "%$search%")->orWhere('total', 'like', "%$search%");
        });
    }

    public function getAll(Request $request): JsonResponse
    {
        $query = Pago::query();
        $pagination = $request->input('pagination', 10);
        $search = $request->input('search');

        if ($search) {
            $query = $this->searchPagos($search);
        }

        $pagos = $query->latest()->paginate($pagination)->appends($request->query());

        return response()->json(
            [
                'pagos' => $pagos,
                'params' => $request->query() ?: null,
            ],
            200,
        );
    }

    public function show($id): JsonResponse
    {
        $pivote_horario_carrera_estudiante = PivoteHorarioCarreraEstudiante::with('pagos')->findOrFail($id);

        if (!$pivote_horario_carrera_estudiante) {
            return response()->json(['message' => 'Pago no encontrado.'], 404);
        }

        return response()->json(
            [
                'pagos' => $pivote_horario_carrera_estudiante->pagos,
                'message' => 'Pago encontrado exitosamente.',
            ],
            200,
        );
    }

    public function save(PagoRequest $request): JsonResponse
    {
        $pago = Pago::create($request->validated());
        $pago->save();

        $pivote_horario_carrera_estudiante = PivoteHorarioCarreraEstudiante::find($request->input('id_pivote_horario_carrera_estudiante'));

        if ($pivote_horario_carrera_estudiante) {
            $pivote_horario_carrera_estudiante->estado_pago = $request->input('estado_pago') ?? 'PENDIENTE';
            $pivote_horario_carrera_estudiante->save();
        } else {
            return response()->json(['message' => 'El pivote relacionado no fue encontrado.'], 404);
        }

        return response()->json(
            [
                'pago' => $pago,
                'message' => 'El pago: ' . $pago->recibo . ', se ha creado exitosamente!.',
            ],
            201,
        );
    }

    public function update($id, PagoRequest $request): JsonResponse
    {
        $pago = Pago::findOrFail($id);

        if (!$pago) {
            return response()->json(['message' => 'El pago' . $request->recibo . ', no encontrado.'], 404);
        }

        $pago->update($request->validated());
        $pago->save();

        $pivote_horario_carrera_estudiante = PivoteHorarioCarreraEstudiante::find($request->input('id_pivote_horario_carrera_estudiante'));

        if ($pivote_horario_carrera_estudiante) {
            $pivote_horario_carrera_estudiante->estado_pago = $request->input('estado_pago') ?? 'PENDIENTE';
            $pivote_horario_carrera_estudiante->save();
        } else {
            return response()->json(['message' => 'El pivote relacionado no fue encontrado.'], 404);
        }

        return response()->json(
            [
                'pago' => $pago,
                'message' => 'El pago: ' . $pago->recibo . ', se ha actualizado exitosamente!.',
            ],
            200,
        );
    }

    public function delete($id): JsonResponse
    {
        $pago = Pago::findOrFail($id);

        if (!$pago) {
            return response()->json(['message' => 'Pago no encontrado.'], 404);
        }

        $pago->delete();

        return response()->json(
            [
                'message' => 'El pago: ' . $pago->recibo . ', se ha eliminado exitosamente!.',
            ],
            200,
        );
    }
}
