<?php

namespace App\Http\Controllers\Finanza;

use App\Http\Controllers\Controller;
use App\Http\Requests\Finanza\BecaRequest;
use App\Models\Finanza\Beca;
use App\Models\PivoteHorarioCarreraEstudiante;
use Illuminate\Http\JsonResponse;
use Illuminate\Http\Request;
use Illuminate\Database\Eloquent\Builder;

class BecaController extends Controller
{
    private function searchBecas(string $search): Builder
    {
        return Beca::where(function ($query) use ($search) {
            $query->where('recibo', 'like', "%$search%")->orWhere('total', 'like', "%$search%");
        });
    }

    public function getAll(Request $request): JsonResponse
    {
        $query = Beca::query();
        $pagination = $request->input('pagination', 10);
        $search = $request->input('search');

        if ($search) {
            $query = $this->searchBecas($search);
        }

        $becas = $query->latest()->paginate($pagination)->appends($request->query());
        
        return response()->json(
            [
                'becas' => $becas,
                'params' => $request->query() ?: null,
            ],
            200,
        );
    }

    public function show($id): JsonResponse
    {
        $pivote_horario_carrera_estudiante = PivoteHorarioCarreraEstudiante::with('becas')->findOrFail($id);
        
        if (!$pivote_horario_carrera_estudiante) {
            return response()->json(['message' => 'Pago no encontrado.'], 404);
        }
        
        return response()->json(
            [
                'becas' => $pivote_horario_carrera_estudiante->becas,
                'message' => 'Pago encontrado exitosamente.',
            ],
            200
        );
    }

    public function save(BecaRequest $request): JsonResponse
    {
        $beca = Beca::create($request->validated());
        $beca->save();
        return response()->json(
            [
                'beca' => $beca,
                'message' => 'El beca: ' . $beca->recibo . ', se ha creado exitosamente!.',
            ],
            201,
        );
    }

    public function update($id, BecaRequest $request): JsonResponse
    {
        $beca = Beca::findOrFail($id);

        if (!$beca) {
            return response()->json(['message' => 'El beca' . $request->recibo . ', no encontrado.'], 404);
        }

        $beca->update($request->validated());
        $beca->save();

        return response()->json(
            [
                'beca' => $beca,
                'message' => 'El beca: ' . $beca->recibo . ', se ha actualizado exitosamente!.',
            ],
            200,
        );
    }

    public function delete($id): JsonResponse
    {
        $beca = Beca::findOrFail($id);

        if (!$beca) {
            return response()->json(['message' => 'Beca no encontrado.'], 404);
        }

        $beca->delete();

        return response()->json(
            [
                'message' => 'El beca: ' . $beca->recibo . ', se ha eliminado exitosamente!.',
            ],
            200,
        );
    }
}
