<?php

namespace App\Http\Controllers\Evaluacion;

use App\Http\Controllers\Controller;
use App\Models\PivoteHorarioCarreraEstudiante;
use App\Services\SearchHorarioCarreraEstudiante;
use Illuminate\Http\JsonResponse;
use Illuminate\Http\Request;

class PromedioController extends Controller
{
  
    public function __construct(SearchHorarioCarreraEstudiante $searchHorarioCarreraEstudiante)
    {
        $this->searchHorarioCarreraEstudiante = $searchHorarioCarreraEstudiante;
    }

    private function getPuntuacionConducta($conducta)
    {
        $puntuaciones = [
            'MUY BUENA' => 7,
            'BUENA' => 6,
            'ACEPTABLE' => 5,
            'REGULAR' => 4,
            'MEJORABLE' => 3,
            'NO COLABORATIVA' => 2,
            'MALA' => 1,
        ];

        return $puntuaciones[$conducta] ?? 0;
    }
    private function getTextoPromedioConducta($promedio)
    {
        if ($promedio >= 7) {
            return 'MUY BUENA';
        } elseif ($promedio > 5 && $promedio <= 6) {
            return 'BUENA';
        } elseif ($promedio > 4 && $promedio <= 5) {
            return 'ACEPTABLE';
        } elseif ($promedio > 3 && $promedio <= 4) {
            return 'REGULAR';
        } elseif ($promedio > 2 && $promedio <= 3) {
            return 'MEJORABLE';
        } elseif ($promedio > 1 && $promedio <= 2) {
            return 'NO COLABORATIVA';
        } elseif ($promedio <= 1) {
            return 'MALA';
        } else {
            return 'NO DISPONIBLE';
        }
    }

    private function getPuntuacionAsistencia($asistencia)
    {
        $puntuaciones = [
            'ASISTIÓ' => 3,
            'TARDE' => 2,
            'FALTO' => 1,
        ];

        return $puntuaciones[$asistencia] ?? 0;
    }

    private function getTextoPromedioAsistencia($promedio)
    {
        if ($promedio > 2 && $promedio <= 3) {
            return 'ASISTE REGULARMENTE';
        } elseif ($promedio > 1 && $promedio <= 2) {
            return 'POR MEJORAR';
        } elseif ($promedio <= 1) {
            return 'NO ASISTE';
        } else {
            return 'NO DISPONIBLE';
        }
    }

    public function getAll(Request $request): JsonResponse
    {
        $query = PivoteHorarioCarreraEstudiante::query();

        $pagination = $request->input('pagination', 10);
        $search = $request->input('search', '');
        $ciclo = $request->input('param.ciclo', 1);
  
        if ($search || $ciclo) {
            $query = $this->searchHorarioCarreraEstudiante->buscarCarreraHorarioDocente($search, $ciclo);
        }

        $promedios = $query->with('estudiante.persona', 'estudiante.colegio', 'estudiante.apoderado.persona', 'carrera', 'pivote_horario.ciclo', 'pivote_horario.horario', 'pagos', 'becas')->latest()->paginate($pagination)->appends($request->query());

        foreach ($promedios as $promedio) {
            $promedioConducta = 0;
            $cantidadConductas = 0;
            $promedioAsistencia = 0;
            $cantidadAsistencias = 0;
            $promedioNotas = 0;
            $cantidadNotas = 0;

            if ($promedio !== null) {
                foreach ($promedio->conductas as $conducta) {
                    $conducta->puntuacion = $this->getPuntuacionConducta($conducta->conducta);
                    $promedioConducta += $conducta->puntuacion;
                    $cantidadConductas++;
                }
                if ($cantidadConductas > 0) {
                    $promedio->promedio_conductas = round($promedioConducta / $cantidadConductas);
                    $promedio->promedio_conductas_texto = $this->getTextoPromedioConducta($promedio->promedio_conductas);
                } else {
                    $promedio->promedio_conductas = null;
                    $promedio->promedio_conductas_texto = null;
                }

                foreach ($promedio->asistencias as $asistencia) {
                    $asistencia->puntuacion = $this->getPuntuacionAsistencia($asistencia->asistencia);
                    $promedioAsistencia += $asistencia->puntuacion;
                    $cantidadAsistencias++;
                }
                if ($cantidadAsistencias > 0) {
                    $promedio->promedio_asistencias = $promedioAsistencia / $cantidadAsistencias;
                    $promedio->promedio_asistencias_texto = $this->getTextoPromedioAsistencia($promedio->promedio_asistencias);
                } else {
                    $promedio->promedio_asistencias = null;
                    $promedio->promedio_asistencias_texto = null;
                }

                foreach ($promedio->notas as $nota) {
                    $promedioNotas += $nota->nota;
                    $cantidadNotas++;
                }
                if ($cantidadNotas > 0) {
                    $promedio->promedio_notas = $promedioNotas / $cantidadNotas;
                } else {
                    $promedio->promedio_notas = null;
                }
            } else {
                $promedio = new \stdClass();
                $promedio->promedio_conductas = null;
                $promedio->promedio_conductas_texto = null;
                $promedio->promedio_asistencias = null;
                $promedio->promedio_asistencias_texto = null;
                $promedio->promedio_notas = null;
            }
        }

        return response()->json(
            [
                'promedios' => $promedios,
                'params' => $request->query() ?: null,
            ],
            200,
        );
    }

    public function show($id): JsonResponse
    {
        $colegio = Colegio::findOrFail($id);

        if (!$colegio) {
            return response()->json(['message' => 'Colegio no encontrado.'], 404);
        }

        return response()->json(
            [
                'colegio' => $colegio,
                'message' => 'Colegio: ' . $colegio->nombre . ', encontrado exitosamente!.',
            ],
            200,
        );
    }

    public function save(ColegioRequest $request): JsonResponse
    {
        $colegio = Colegio::create($request->validated());
        $colegio->save();
        return response()->json(
            [
                'colegio' => $colegio,
                'message' => 'El colegio: ' . $colegio->nombre . ', se ha creado exitosamente!.',
            ],
            201,
        );
    }

    public function update($id, ColegioRequest $request): JsonResponse
    {
        $colegio = Colegio::findOrFail($id);

        if (!$colegio) {
            return response()->json(['message' => 'El colegio' . $request->nombre . ', no encontrado.'], 404);
        }

        $colegio->update($request->validated());
        $colegio->save();

        return response()->json(
            [
                'colegio' => $colegio,
                'message' => 'El colegio: ' . $colegio->nombre . ', se ha actualizado exitosamente!.',
            ],
            200,
        );
    }

    public function delete($id): JsonResponse
    {
        $colegio = Colegio::findOrFail($id);

        if (!$colegio) {
            return response()->json(['message' => 'Colegio no encontrado.'], 404);
        }

        $colegio->delete();

        return response()->json(
            [
                'message' => 'El colegio: ' . $colegio->nombre . ', se ha eliminado exitosamente!.',
            ],
            200,
        );
    }
}
