<?php

namespace App\Http\Controllers\Evaluacion;

use App\Http\Controllers\Controller;
use App\Http\Requests\Evaluacion\PreguntaRequest;
use App\Models\Evaluacion\Pregunta;
use App\Models\PivotePreguntaDocente;
use Illuminate\Database\Eloquent\Builder;
use Illuminate\Http\JsonResponse;
use Illuminate\Http\Request;

class PreguntaController extends Controller
{
    private function searchPreguntas(string $search): Builder
    {
        return Pregunta::where(function ($query) use ($search) {
            $query->where('pregunta', 'like', "%$search%")->orWhere('puntuacion', 'like', "%$search%");
        });
    }

    public function getAll(Request $request): JsonResponse
    {
        $query = Pregunta::query();
        $pagination = $request->input('pagination', 10);
        $search = $request->input('search');

        if ($search) {
            $query = $this->searchPreguntas($search);
        }

        $preguntas = $query->with('pivote_preguntas_docentes.pivote_encuestas_docentes', 'pivote_preguntas_docentes.pregunta', 'pivote_preguntas_docentes.pivote_area_curso_docente.docente.persona')->latest()->paginate($pagination);

        return response()->json(
            [
                'preguntas' => $preguntas,
                'params' => $request->query() ?: null,
            ],
            200,
        );
    }

    public function show($id): JsonResponse
    {
        $pregunta = Pregunta::findOrFail($id);

        if (!$pregunta) {
            return response()->json(['message' => 'Pregunta no encontrado.'], 404);
        }

        return response()->json(
            [
                'pregunta' => $pregunta,
                'message' => 'Pregunta: ' . $pregunta->pregunta . ', encontrado exitosamente!.',
            ],
            200,
        );
    }

    public function save(PreguntaRequest $request): JsonResponse
    {
        $pregunta = Pregunta::create($request->validated());
        $pregunta->save();

        $docentes = $request->docentes;

        foreach ($docentes as $docente) {
            PivotePreguntaDocente::create([
                'id_pregunta' => $pregunta->id,
                'id_pivote_area_curso_docente' => $docente['id'],
            ]);
        }

        return response()->json(
            [
                'pregunta' => $pregunta,
                'message' => 'El pregunta: ' . $pregunta->pregunta . ', se ha creado exitosamente!.',
            ],
            201,
        );
    }

    public function update($id, PreguntaRequest $request): JsonResponse
    {
        $pregunta = Pregunta::findOrFail($id);

        if (!$pregunta) {
            return response()->json(['message' => 'La pregunta ' . $request->pregunta . ' no fue encontrada.'], 404);
        }

        $docentesActuales = $pregunta->pivote_preguntas_docentes->pluck('id_pivote_area_curso_docente')->toArray();

        $docentesNuevos = collect($request->docentes)->pluck('id')->toArray();

        $docentesAEliminar = array_diff($docentesActuales, $docentesNuevos);

        $docentesAAgregar = array_diff($docentesNuevos, $docentesActuales);

        if (!empty($docentesAEliminar)) {
            PivotePreguntaDocente::where('id_pregunta', $pregunta->id)
                ->whereIn('id_pivote_area_curso_docente', $docentesAEliminar)
                ->delete();
        }

        foreach ($docentesAAgregar as $idDocente) {
            PivotePreguntaDocente::create([
                'id_pregunta' => $pregunta->id,
                'id_pivote_area_curso_docente' => $idDocente,
            ]);
        }

        $pregunta->update($request->validated());

        return response()->json([
            'pregunta' => $pregunta->load('pivote_preguntas_docentes'),
            'message' => 'La pregunta: ' . $pregunta->pregunta . ' se ha actualizado exitosamente!',
        ], 200);
    }


    public function delete($id): JsonResponse
    {
        $pregunta = Pregunta::findOrFail($id);

        if (!$pregunta) {
            return response()->json(['message' => 'Pregunta no encontrada.'], 404);
        }

        $pregunta->pivote_preguntas_docentes()->delete();

        $pregunta->delete();

        return response()->json(
            [
                'message' => 'La pregunta: ' . $pregunta->pregunta . ', se ha eliminado exitosamente!',
            ],
            200,
        );
    }
}
