<?php

namespace App\Http\Controllers\Evaluacion;

use App\Http\Controllers\Controller;
use App\Http\Requests\Evaluacion\NotaRequest;
use App\Models\Evaluacion\Nota;
use Illuminate\Database\Eloquent\Builder;
use Illuminate\Http\JsonResponse;
use Illuminate\Http\Request;

class NotaController extends Controller
{

    private function applyFilters(Request $request): Builder
    {
        $query = Nota::query();

        if ($search = $request->input('search')) {
            $query->where('nota', 'like', "%$search%");
        }

        if ($param = $request->input('param.id_pivote_horario_carrera_estudiante')) {
            $query->where('id_pivote_horario_carrera_estudiante', $param);
        }

        if ($param = $request->input('param.tipo')) {
            $query->where('tipo', $param);
        }

        return $query;
    }
    
    public function getAll(Request $request): JsonResponse
    {
        $pagination = $request->input('pagination', 10);
        $query = $this->applyFilters($request);
        $notas = $query->latest()->paginate($pagination)->appends($request->query());

        return response()->json(
            [
                'notas' => $notas,
                'params' => $request->query() ?: null,
            ],
            200,
        );
    }

    public function show($id): JsonResponse
    {
        $nota = Nota::findOrFail($id);

        if (!$nota) {
            return response()->json(['message' => 'Nota no encontrado.'], 404);
        }

        return response()->json(
            [
                'nota' => $nota,
                'message' => 'Nota: ' . $nota->name . ', encontrado exitosamente!.',
            ],
            200,
        );
    }

    public function save(NotaRequest $request): JsonResponse
    {
        $nota = Nota::create($request->validated());
        $nota->save();
        return response()->json(
            [
                'nota' => $nota,
                'message' => 'El nota: ' . $nota->name . ', se ha creado exitosamente!.',
            ],
            201,
        );
    }

    public function update($id, NotaRequest $request): JsonResponse
    {
        $nota = Nota::findOrFail($id);

        if (!$nota) {
            return response()->json(['message' => 'El nota' . $request->name . ', no encontrado.'], 404);
        }

        $nota->update($request->validated());
        $nota->save();

        return response()->json(
            [
                'nota' => $nota,
                'message' => 'El nota: ' . $nota->name . ', se ha actualizado exitosamente!.',
            ],
            200,
        );
    }

    public function delete($id): JsonResponse
    {
        $nota = Nota::findOrFail($id);

        if (!$nota) {
            return response()->json(['message' => 'Nota no encontrado.'], 404);
        }

        $nota->delete();

        return response()->json(
            [
                'message' => 'El nota: ' . $nota->name . ', se ha eliminado exitosamente!.',
            ],
            200,
        );
    }
}
