<?php

namespace App\Http\Controllers\Evaluacion;

use App\Http\Controllers\Controller;
use App\Http\Requests\Evaluacion\EncuestaRequest;
use App\Models\Evaluacion\Encuesta;
use App\Models\PivoteEncuestaDocente;
use App\Models\PivotePreguntaDocente;
use Illuminate\Database\Eloquent\Builder;
use Illuminate\Http\JsonResponse;
use Illuminate\Http\Request;

class EncuestaController extends Controller
{
    private function searchEncuestas(string $search): Builder
    {
        return Encuesta::where(function ($query) use ($search) {
            $query->where('nombre', 'like', "%$search%");
        });
    }

    public function getAll(Request $request): JsonResponse
    {
        $query = Encuesta::query();
        $pagination = $request->input('pagination', 10);
        $search = $request->input('search');

        if ($search) {
            $query = $this->searchEncuestas($search);
        }

        $encuestas = $query->with('pivote_encuestas_docentes.pivote_pregunta_docente.pivote_area_curso_docente.docente.persona','pivote_encuestas_docentes.pivote_pregunta_docente.pregunta')->latest()->paginate($pagination);
        
        return response()->json(
            [
                'encuestas' => $encuestas,
                'params' => $request->query() ?: null,
            ],
            200,
        );
    }

    public function show($id): JsonResponse
    {
        $encuesta = Encuesta::findOrFail($id);

        if (!$encuesta) {
            return response()->json(['message' => 'Encuesta no encontrado.'], 404);
        }

        return response()->json(
            [
                'encuesta' => $encuesta,
                'message' => 'Encuesta: ' . $encuesta->nombre . ', encontrado exitosamente!.',
            ],
            200,
        );
    }

    public function save(EncuestaRequest $request): JsonResponse
    {
        $encuesta = Encuesta::create($request->validated());

        $docentes = $request->input('preguntasDocentes', []);
        if (!is_array($docentes)) {
            return response()->json(['message' => 'El campo docentes debe ser un array válido.'], 422);
        }

        foreach ($docentes as $docente) {
            PivoteEncuestaDocente::create([
                'id_encuesta' => $encuesta->id,
                'id_pivote_pregunta_docente' => $docente['id']
            ]);
        }

        return response()->json(
            [
                'encuesta' => $encuesta,
                'message' => 'La encuesta: ' . $encuesta->nombre . ', se ha creado exitosamente!',
            ],
            201,
        );
    }

    public function update($id, EncuestaRequest $request): JsonResponse
    {
        $encuesta = Encuesta::findOrFail($id);

        if (!$encuesta) {
            return response()->json(['message' => 'La encuesta ' . $request->nombre . ' no fue encontrada.'], 404);
        }

        $docentes = $request->input('preguntasDocentes', []);

        if (!is_array($docentes)) {
            return response()->json(['message' => 'El campo docentes debe ser un array válido.'], 422);
        }

        $encuesta->update($request->validated());
        $encuesta->pivote_encuestas_docentes()->delete();

        foreach ($docentes as $docente) {
            PivoteEncuestaDocente::create([
                'id_encuesta' => $encuesta->id,
                'id_pivote_pregunta_docente' => $docente['id']
            ]);
        }

        return response()->json(
            [
                'encuesta' => $encuesta,
                'message' => 'La encuesta: ' . $encuesta->nombre . ' se ha actualizado exitosamente.',
            ],
            200,
        );
    }

    public function delete($id): JsonResponse
    {
        $encuesta = Encuesta::findOrFail($id);

        if (!$encuesta) {
            return response()->json(['message' => 'Encuesta no encontrado.'], 404);
        }
        $encuesta->pivote_encuestas_docentes()->delete();
        $encuesta->delete();

        return response()->json(
            [
                'message' => 'El encuesta: ' . $encuesta->nombre . ', se ha eliminado exitosamente!.',
            ],
            200,
        );
    }
}
