<?php

namespace App\Http\Controllers\Educacion;

use App\Http\Controllers\Controller;
use App\Http\Requests\Educacion\UniversidadRequest;
use App\Models\Educacion\Universidad;
use Illuminate\Http\Request;
use Illuminate\Http\JsonResponse;
use Illuminate\Database\Eloquent\Builder;

class UniversidadController extends Controller
{
    private function searchUniversidades(string $search): Builder
    {
        return Universidad::query()
            ->where('nombre', 'like', "%{$search}%")
            ->orWhere('tipo', 'like', "%{$search}%");
    }

    public function getAll(Request $request): JsonResponse
    {
        $query = Universidad::query();
        $pagination = $request->input('pagination', 10);
        $search = $request->input('search');

        if ($search) {
            $query = $this->searchUniversidades($search);
        }
        $universidades = $query->with('pivote_horarios_carreras_estudiantes')->latest()->paginate($pagination)->appends($request->query());

        return response()->json(
            [
                'universidades' => $universidades,
                'params' => $request->query() ?: null,
            ],
            200,
        );
    }

    public function show($id): JsonResponse
    {
        $universidad = Universidad::findOrFail($id);

        if (!$universidad) {
            return response()->json(['message' => 'Universidad no encontrado.'], 404);
        }

        return response()->json(
            [
                'universidad' => $universidad,
                'message' => 'El área: ' . $universidad->nombre . ', encontrado exitosamente!.',
            ],
            200,
        );
    }

    public function save(UniversidadRequest $request): JsonResponse
    {
        $universidad = Universidad::create($request->validated());
        $universidad->save();
        return response()->json(
            [
                'universidad' => $universidad,
                'message' => 'El área: ' . $universidad->nombre . ', se ha creado exitosamente!.',
            ],
            201,
        );
    }

    public function update($id, UniversidadRequest $request): JsonResponse
    {
        $universidad = Universidad::findOrFail($id);

        if (!$universidad) {
            return response()->json(['message' => 'El área' . $request->nombre . ', no encontrado.'], 404);
        }

        $universidad->update($request->validated());
        $universidad->save();

        return response()->json(
            [
                'universidad' => $universidad,
                'message' => 'El área: ' . $universidad->nombre . ', se ha actualizado exitosamente!.',
            ],
            200,
        );
    }

    public function delete($id): JsonResponse
    {
        $universidad = Universidad::findOrFail($id);

        if (!$universidad) {
            return response()->json(['message' => 'Universidad no encontrado.'], 404);
        }

        $universidad->delete();

        return response()->json(
            [
                'message' => 'El área: ' . $universidad->nombre . ', se ha eliminado exitosamente!.',
            ],
            200,
        );
    }
}
