<?php

namespace App\Http\Controllers\Educacion;

use App\Http\Controllers\Controller;
use App\Http\Requests\Educacion\ColegioRequest;
use App\Models\Educacion\Colegio;
use Illuminate\Database\Eloquent\Builder;
use Illuminate\Http\JsonResponse;
use Illuminate\Http\Request;

class ColegioController extends Controller
{
    private function searchColegios(string $search): Builder
    {
        return Colegio::where(function ($query) use ($search) {
            $query
                ->where('nombre', 'like', "%$search%")
                ->orWhere('iniciales', 'like', "%$search%")
                ->orWhere('tipo', 'like', "%$search%");
        });
    }

    private function filterByTipo(string $tipo): Builder
    {
        return Colegio::where('tipo', $tipo);
    }

    public function getAll(Request $request): JsonResponse
    {
        $query = Colegio::query();
        $pagination = $request->input('pagination', 10);
        $tipo = $request->input('tipo');
        $search = $request->input('search');

        if ($search) {
            $query = $this->searchColegios($search);
        }

        if ($tipo) {
            $query = $this->filterByTipo($tipo);
        }

        $colegios = $query->with('estudiantes')->latest()->paginate($pagination)->appends($request->query());

        return response()->json(
            [
                'colegios' => $colegios,
                'params' => $request->query() ?: null,
            ],
            200,
        );
    }

    public function show($id): JsonResponse
    {
        $colegio = Colegio::findOrFail($id);

        if (!$colegio) {
            return response()->json(['message' => 'Colegio no encontrado.'], 404);
        }

        return response()->json(
            [
                'colegio' => $colegio,
                'message' => 'Colegio: ' . $colegio->nombre . ', encontrado exitosamente!.',
            ],
            200,
        );
    }

    public function save(ColegioRequest $request): JsonResponse
    {
        $colegio = Colegio::create($request->validated());
        $colegio->save();
        return response()->json(
            [
                'colegio' => $colegio,
                'message' => 'El colegio: ' . $colegio->nombre . ', se ha creado exitosamente!.',
            ],
            201,
        );
    }

    public function update($id, ColegioRequest $request): JsonResponse
    {
        $colegio = Colegio::findOrFail($id);

        if (!$colegio) {
            return response()->json(['message' => 'El colegio' . $request->nombre . ', no encontrado.'], 404);
        }

        $colegio->update($request->validated());
        $colegio->save();

        return response()->json(
            [
                'colegio' => $colegio,
                'message' => 'El colegio: ' . $colegio->nombre . ', se ha actualizado exitosamente!.',
            ],
            200,
        );
    }

    public function delete($id): JsonResponse
    {
        $colegio = Colegio::findOrFail($id);

        if (!$colegio) {
            return response()->json(['message' => 'Colegio no encontrado.'], 404);
        }

        $colegio->delete();

        return response()->json(
            [
                'message' => 'El colegio: ' . $colegio->nombre . ', se ha eliminado exitosamente!.',
            ],
            200,
        );
    }
}
