<?php

namespace App\Http\Controllers\Auth;

use App\Http\Controllers\Controller;
use App\Http\Requests\Auth\UserRequest;
use App\Models\User;
use Illuminate\Database\Eloquent\Builder;
use Illuminate\Http\JsonResponse;
use Illuminate\Http\Request;

class UserController extends Controller
{
    private function searchUsers(Builder $query, string $search): Builder
    {
        return $query->where(function ($subQuery) use ($search) {
            $subQuery
                ->where('name', 'like', "%$search%")
                ->orWhere('email', 'like', "%$search%")
                ->orWhere('rol', 'like', "%$search%");
        });
    }

    private function filterByRole(Builder $query, string $role): Builder
    {
        return $query->where('rol', $role);
    }

    private function filterByRoles(Builder $query, array $roles): Builder
{
    return $query->whereIn('rol', $roles);
}

public function getAll(Request $request): JsonResponse
{
    $query = User::query();
    $pagination = $request->input('pagination', 10);
    $role = $request->input('rol');
    $roles = $request->input('param.roles');
    $search = $request->input('search');

    if ($search) {
        $query = $this->searchUsers($query, $search);
    }

    if ($role) {
        $query = $this->filterByRole($query, $role);
    }

    if ($roles && is_array($roles)) {
        $query = $this->filterByRoles($query, $roles);
    }

    $usuarios = $query->latest()->paginate($pagination);

    return response()->json(
        [
            'usuarios' => $usuarios,
            'params' => $request->query() ?: null,
        ],
        200
    );
}


    public function show($id): JsonResponse
    {
        $usuario = User::findOrFail($id);

        if (!$usuario) {
            return response()->json(['message' => 'Usuario no encontrado.'], 404);
        }

        return response()->json(
            [
                'usuario' => $usuario,
                'message' => 'Usuario: ' . $usuario->name . ', encontrado exitosamente!.',
            ],
            200,
        );
    }

    public function save(UserRequest $request): JsonResponse
    {
        $usuario = User::create($request->validated());
        $usuario->save();
        return response()->json(
            [
                'usuario' => $usuario,
                'message' => 'El usuario: ' . $usuario->name . ', se ha creado exitosamente!.',
            ],
            201,
        );
    }

    public function update($id, UserRequest $request): JsonResponse
    {
        $usuario = User::findOrFail($id);

        if (!$usuario) {
            return response()->json(['message' => 'El usuario' . $request->name . ', no encontrado.'], 404);
        }

        $usuario->update($request->validated());
        $usuario->save();

        return response()->json(
            [
                'usuario' => $usuario,
                'message' => 'El usuario: ' . $usuario->name . ', se ha actualizado exitosamente!.',
            ],
            200,
        );
    }

    public function delete($id): JsonResponse
    {
        $usuario = User::findOrFail($id);

        if (!$usuario) {
            return response()->json(['message' => 'Usuario no encontrado.'], 404);
        }

        $usuario->delete();

        return response()->json(
            [
                'message' => 'El usuario: ' . $usuario->name . ', se ha eliminado exitosamente!.',
            ],
            200,
        );
    }
}
