<?php

namespace App\Http\Controllers\Academico;

use App\Http\Controllers\Controller;
use App\Http\Requests\Academico\HorarioRequest;
use App\Models\Academico\Horario;
use Illuminate\Http\JsonResponse;
use Illuminate\Http\Request;
use App\Services\SearchHorario;

class HorarioController extends Controller
{
    protected $searchHorario;

    public function __construct(SearchHorario $searchHorario)
    {
        $this->searchHorario = $searchHorario;
    }

    public function getAll(Request $request): JsonResponse
    {
        $query = Horario::query();
        $pagination = $request->input('pagination', 10);
        $search = $request->input('search');

        if ($search) {
            $query = $this->searchHorario->buscarHorarios($search);
        }

        $horarios = $query->with('pivote_horarios')->latest()->paginate($pagination)->appends($request->query());

        return response()->json(
            [
                'horarios' => $horarios,
                'params' => $request->query() ?: null,
            ],
            200,
        );
    }

    public function show($id): JsonResponse
    {
        $horario = Horario::findOrFail($id);

        if (!$horario) {
            return response()->json(['message' => 'Horario no encontrado.'], 404);
        }

        return response()->json(
            [
                'horario' => $horario,
                'message' => 'Horario: ' . $horario->nombre . ', encontrado exitosamente!.',
            ],
            200,
        );
    }

    public function save(HorarioRequest $request): JsonResponse
    {
        $horario = Horario::create($request->validated());
        $horario->save();
        return response()->json(
            [
                'horario' => $horario,
                'message' => 'El horario: ' . $horario->nombre . ', se ha creado exitosamente!.',
            ],
            201,
        );
    }

    public function update($id, HorarioRequest $request): JsonResponse
    {
        $horario = Horario::findOrFail($id);

        if (!$horario) {
            return response()->json(['message' => 'El horario' . $request->nombre . ', no encontrado.'], 404);
        }

        $horario->update($request->validated());
        $horario->save();

        return response()->json(
            [
                'horario' => $horario,
                'message' => 'El horario: ' . $horario->nombre . ', se ha actualizado exitosamente!.',
            ],
            200,
        );
    }

    public function delete($id): JsonResponse
    {
        $horario = Horario::findOrFail($id);

        if (!$horario) {
            return response()->json(['message' => 'Horario no encontrado.'], 404);
        }

        $horario->delete();

        return response()->json(
            [
                'message' => 'El horario: ' . $horario->nombre . ', se ha eliminado exitosamente!.',
            ],
            200,
        );
    }
}
