<?php

namespace App\Http\Controllers\Academico;

use App\Http\Controllers\Controller;
use App\Http\Requests\Academico\CursoRequest;
use App\Models\Academico\Curso;
use Illuminate\Database\Eloquent\Builder;
use Illuminate\Http\JsonResponse;
use Illuminate\Http\Request;

class CursoController extends Controller
{
    public function searchCursos(string $search): Builder
    {
        return Curso::where(function ($query) use ($search) {
            $query->where('nombre', 'like', "%$search%");
        });
    }

    public function getAll(Request $request): JsonResponse
    {
        $query = Curso::query();
        $pagination = $request->input('pagination', 10);
        $search = $request->input('search');

        if ($search) {
            $query = $this->searchCursos($search);
        }

        $cursos = $query->with('pivote_areas_cursos_docentes.docente.persona', 'pivote_areas_cursos_docentes.area')->latest()->paginate($pagination)->appends($request->query());

        return response()->json(
            [
                'cursos' => $cursos,
                'params' => $request->query() ?: null,
            ],
            200,
        );
    }

    public function show($id): JsonResponse
    {
        $curso = Curso::findOrFail($id);

        if (!$curso) {
            return response()->json(['message' => 'Curso no encontrado.'], 404);
        }

        return response()->json(
            [
                'curso' => $curso,
                'message' => 'Curso: ' . $curso->nombre . ', encontrado exitosamente!.',
            ],
            200,
        );
    }

    public function save(CursoRequest $request): JsonResponse
    {
        $curso = Curso::create($request->validated());
        $curso->save();
        return response()->json(
            [
                'curso' => $curso,
                'message' => 'El curso: ' . $curso->nombre . ', se ha creado exitosamente!.',
            ],
            201,
        );
    }

    public function update($id, CursoRequest $request): JsonResponse
    {
        $curso = Curso::findOrFail($id);

        if (!$curso) {
            return response()->json(['message' => 'El curso' . $request->nombre . ', no encontrado.'], 404);
        }

        $curso->update($request->validated());
        $curso->save();

        return response()->json(
            [
                'curso' => $curso,
                'message' => 'El curso: ' . $curso->nombre . ', se ha actualizado exitosamente!.',
            ],
            200,
        );
    }

    public function delete($id): JsonResponse
    {
        $curso = Curso::findOrFail($id);

        if (!$curso) {
            return response()->json(['message' => 'Curso no encontrado.'], 404);
        }

        $curso->pivote_areas_cursos_docentes()->delete();

        $curso->delete();

        return response()->json(
            [
                'message' => 'El curso: ' . $curso->nombre . ', y todas sus relaciones asociadas se han eliminado exitosamente.',
            ],
            200,
        );
    }
}
